package ai.apiverse.apisuite.mirror.agent.config;

import ai.apiverse.apisuite.mirror.agent.ApimonitorHTTPConnection;
import ai.apiverse.apisuite.mirror.agent.SDKLogger;
import ai.apiverse.apisuite.mirror.agent.SDKVersion;

import ai.apiverse.apisuite.mirror.models.data.AgentConfig;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import org.springframework.scheduling.concurrent.CustomizableThreadFactory;

import java.util.*;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

public final class SimpleConfigManager implements ConfigManager {
    private final String ctUrl;
    private final ApimonitorHTTPConnection apimonitorHTTPConnection;
    private final String userApplicationName;
    private ScheduledExecutorService executorService;

    private SDKLogger logger;
    private final List<ConfigUpdateListener> configUpdateListeners = Collections.synchronizedList(new LinkedList<>());

    private final Map<String, String> configHeader;
    private final String agentId;

    public SimpleConfigManager(final String ctUrl,
                               final String userApplicationName,
                               final String agentId,
                               ApimonitorHTTPConnection apimonitorHTTPConnection,
                               SDKLogger logger) {
        this.ctUrl = ctUrl;
        this.userApplicationName = userApplicationName;
        this.agentId = agentId;
        this.apimonitorHTTPConnection = apimonitorHTTPConnection;
        this.logger = logger;
        configHeader = new HashMap<>();
        configHeader.put(SDKVersion.MAJOR_VERSION_KEY, SDKVersion.MAJOR_VERSION);
        configHeader.put(SDKVersion.MINOR_VERSION_KEY, SDKVersion.MINOR_VERSION);
    }

    public boolean subscribeToUpdates(ConfigUpdateListener configUpdateListener) {
        configUpdateListeners.add(configUpdateListener);
        return true;
    }

    public synchronized boolean shutdown() {
        logger.info("Shutting down SimpleConfigManager");
        if (null == executorService) {
            logger.info("Shutting down executorService");
            return true;
        }
        if (executorService.isShutdown()) {
            return true;
        }
        try {
            executorService.shutdown(); // Disable new tasks from being submitted
            if (!executorService.awaitTermination(10, TimeUnit.SECONDS)) {
                executorService.shutdownNow(); // Cancel currently executing tasks
                if (!executorService.awaitTermination(10, TimeUnit.SECONDS)) {
                    logger.error("Still not able to shutdown SimpleConfigManager's executorService");
                }
            }
        } catch (InterruptedException e) {
            logger.error("Error while shutting down SimpleConfigManager's executorService", e);
        }
        return executorService.isShutdown();

    }

    public synchronized boolean init () {
        try {
            executorService = Executors.newSingleThreadScheduledExecutor(new CustomizableThreadFactory("config-"));
            scheduleConfigRefresh(60);
        } catch (Exception e) {
            logger.error("Error in SimpleConfigManager's init()", e);
            return false;
        }
        return true;
    }

    private void scheduleConfigRefresh(int timeInSecs) {
        Runnable runnable = () -> {
            try {
                ConfigOrError configOrError = fetchConfig();
                if (null != configOrError.errorCode) {
                    //scheduleConfigRefresh(60);
                    onUnSuccessfulConfigFetch(configOrError.errorCode);
                } else {
                    scheduleConfigRefresh(configOrError.getAgentConfig().getConfigFetchFreqInSec());
                    onSuccessfulConfigFetch(configOrError.agentConfig);
                }
            } catch (Exception e) {
                //scheduleConfigRefresh(60);
                logger.error("Error in SimpleConfigManager's scheduleConfigRefresh()", e);
            }
        };
        executorService.schedule(runnable, timeInSecs, TimeUnit.SECONDS);
    }

    private ConfigOrError fetchConfig() {
        try {
            Map<String, String> queryParams = new HashMap<>();
            queryParams.put("appName", userApplicationName);
            queryParams.put("agentId", agentId);
            AgentConfig agentConfig = this.apimonitorHTTPConnection.getSDKConfig();
            if (null == agentConfig) {
                logger.error("Received null config :");
                return new ConfigOrError(null, ConfigOrError.ErrorCode.INVALID_CONFIG);
            }
            if (AgentConfigUtils.isConfigValid(agentConfig)) {
                return new ConfigOrError(agentConfig, null);
            } else {
                logger.error("Received invalid config :" + agentConfig);
                return new ConfigOrError(null, ConfigOrError.ErrorCode.INVALID_CONFIG);
            }

        } catch (Exception exception) {
            logger.error("Error while fetching config", exception);
            return new ConfigOrError(null, ConfigOrError.ErrorCode.PARSE_ERROR);
        }
    }

    private void onSuccessfulConfigFetch(AgentConfig agentConfig) {
        for (ConfigUpdateListener configUpdateListener: configUpdateListeners) {
            configUpdateListener.onSuccessfulConfigUpdate(agentConfig);
        }
    }

    private void onUnSuccessfulConfigFetch(ConfigOrError.ErrorCode errorCode) {
        for (ConfigUpdateListener configUpdateListener: configUpdateListeners) {
            configUpdateListener.onErroneousConfigUpdate();
        }
    }
    @RequiredArgsConstructor
    @Getter
    @Setter
    private static class ConfigOrError {
        private final AgentConfig agentConfig;
        private final ErrorCode errorCode;

        private enum ErrorCode {
            TIMEOUT,
            PARSE_ERROR,
            INVALID_CONFIG
        }
    }



}
