package ai.apiverse.apisuite.scanner;

import ai.apiverse.apisuite.mirror.agent.buffer.Constants;
import ai.apiverse.apisuite.mirror.models.constant.HttpRequestMethod;
import ai.apiverse.apisuite.mirror.models.data.AgentConfig;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializer;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.MethodParameter;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.mvc.method.RequestMappingInfo;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

import java.io.OutputStream;
import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Parameter;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.util.*;


@Service
public class ApiScannerConfig {
    @Autowired EndpointService endpointService;
    Gson gson = getGsonObject();

    @Value("${apimonitor.applicationName:}")
    private String userApplicationName;

    private Gson getGsonObject() {
        Gson gson = new GsonBuilder()
                .registerTypeAdapter(LocalDate.class, (JsonSerializer<LocalDate>) (src, typeOfSrc, context) -> new JsonPrimitive(src.format(DateTimeFormatter.ISO_LOCAL_DATE)))
                .registerTypeAdapter(LocalTime.class, (JsonSerializer<LocalTime>) (src, typeOfSrc, context) -> new JsonPrimitive(src.format(DateTimeFormatter.ISO_LOCAL_TIME)))
                .registerTypeAdapter(LocalDateTime.class, (JsonSerializer<LocalDateTime>) (src, typeOfSrc, context) -> new JsonPrimitive(src.format(DateTimeFormatter.ISO_LOCAL_DATE_TIME)))
                .registerTypeAdapter(Instant.class, (JsonSerializer<Instant>) (src, typeOfSrc, context) -> new JsonPrimitive(src.toString()))
                .registerTypeAdapter(ZonedDateTime.class, (JsonSerializer<ZonedDateTime>) (src, typeOfSrc, context) -> new JsonPrimitive(src.format(DateTimeFormatter.ISO_ZONED_DATE_TIME)))
                .registerTypeAdapter(OffsetDateTime.class, (JsonSerializer<OffsetDateTime>) (src, typeOfSrc, context) -> new JsonPrimitive(src.format(DateTimeFormatter.ISO_OFFSET_DATE_TIME)))
                .registerTypeAdapter(OffsetTime.class, (JsonSerializer<OffsetTime>) (src, typeOfSrc, context) -> new JsonPrimitive(src.format(DateTimeFormatter.ISO_OFFSET_TIME)))
                .registerTypeAdapter(Year.class, (JsonSerializer<Year>) (src, typeOfSrc, context) -> new JsonPrimitive(src.toString()))
                .registerTypeAdapter(YearMonth.class, (JsonSerializer<YearMonth>) (src, typeOfSrc, context) -> new JsonPrimitive(src.format(DateTimeFormatter.ofPattern("yyyy-MM"))))
                .registerTypeAdapter(MonthDay.class, (JsonSerializer<MonthDay>) (src, typeOfSrc, context) -> new JsonPrimitive(src.format(DateTimeFormatter.ofPattern("MM-dd"))))
                .registerTypeAdapter(Duration.class, (JsonSerializer<Duration>) (src, typeOfSrc, context) -> new JsonPrimitive(src.toString()))
                .registerTypeAdapter(Period.class, (JsonSerializer<Period>) (src, typeOfSrc, context) -> new JsonPrimitive(src.toString()))
                .serializeNulls()
                .create();
        return gson;
    }

    // @Bean
    public Object apiScanner(RequestMappingHandlerMapping requestMappingHandlerMapping) {
        // You may wish to do this at some other time or trigger it in some other way
        List<EndpointInfo> endpoints = new ArrayList<>();
        Map<Class<?>,List<EndpointInfo>> controllerVsApis = new HashMap<>();
        requestMappingHandlerMapping.getHandlerMethods().forEach((info, method) -> {
            EndpointInfo endpointInfo = getMappingInfo(info, method);
            Class<?> controller = method.getBeanType();
            List<EndpointInfo> apilist = controllerVsApis.getOrDefault(controller, new ArrayList<>());
            apilist.add(endpointInfo);
            controllerVsApis.put(controller,apilist);
            endpoints.add(endpointInfo);
        });
        System.out.println("Scanned Endpoints:");
        endpoints.forEach(System.out::println);
        return controllerVsApis;

    }

    EndpointInfo getMappingInfo(RequestMappingInfo info, HandlerMethod method) {
        EndpointInfo endpointInfo = new EndpointInfo();

        try {
            // Extract the URL pattern (assuming only one)
            endpointInfo.setPattern(info.getPathPatternsCondition().getPatterns().iterator().next().getPatternString());

            // Extract the HTTP method (assuming only one)
            Set<RequestMethod> methods = info.getMethodsCondition().getMethods();
            if (methods != null && !methods.isEmpty()) {
                endpointInfo.setMethod(methods.iterator().next().name());
            }

            // Extract details about the method
            endpointInfo.setMethodDetails(method.getMethod().toString());

            // Extract return type
            endpointInfo.setReturnParameter(method.getReturnType().getParameterType().getName());

            // Extract input parameters
            Map<Object, Object> inputParameters = new HashMap<>();

            for (MethodParameter parameter : method.getMethodParameters()) {
                String type = getParameterType(parameter);
                Parameter nativeParameter = parameter.getExecutable().getParameters()[parameter.getParameterIndex()];
                String name = nativeParameter.getName();
                String paramDto = nativeParameter.getType().getSimpleName().toString();
                String qualifiedName = nativeParameter.getType().getName();
                Map<String, String> details = new HashMap<>();
                details.put("dataType", paramDto);
                details.put("name", name);
                details.put("type", type);
                details.put("class",qualifiedName);
                details.put("sampleJson",sampleJson(qualifiedName,name));
                inputParameters.put(name, details);
            }
            endpointInfo.setInputParameter(inputParameters);

        } catch (Exception e) {
            // Log the error and details for debugging
            System.err.println("Error while processing endpoint info for method: " + method.getMethod().getName());
            e.printStackTrace();

            // Return a null or default object if needed
            // Depending on your specific requirements, you might want to handle this differently
            return null;
        }

        return endpointInfo;
    }
    private String sampleJson(String className,String variableName) {
        Class<?> clazz;
        Object instance = null;
        try {
            clazz = Class.forName(className);
            if (clazz.isEnum()) {
                Object[] enumConstants = clazz.getEnumConstants();
                instance = enumConstants.length > 0 ? enumConstants[0] : null;
            } else if (clazz.isPrimitive() || clazz == Integer.class || clazz == Long.class || clazz == Short.class
                    || clazz == Double.class || clazz == Float.class || clazz == Byte.class || clazz == Character.class
                    || clazz == Boolean.class) {
                instance = Array.get(Array.newInstance(clazz, 1), 0);
            } else if (clazz == String.class) {
                instance = "";
            } else if (clazz == LocalDateTime.class) {
                instance = LocalDateTime.now();
            } else if (clazz == LocalDate.class) {
                instance = LocalDate.now();
            } else if (clazz == LocalTime.class) {
                instance = LocalTime.now();
            } else if (clazz == Instant.class) {
                instance = Instant.now();
            } else if (clazz == ZonedDateTime.class) {
                instance = ZonedDateTime.now();
            } else if (clazz == OffsetDateTime.class) {
                instance = OffsetDateTime.now();
            } else if (clazz == OffsetTime.class) {
                instance = OffsetTime.now();
            } else if (clazz == YearMonth.class) {
                instance = YearMonth.now();
            } else if (clazz == Year.class) {
                instance = Year.now();
            } else if (clazz == MonthDay.class) {
                instance = MonthDay.now();
            } else if (clazz == Duration.class) {
                instance = Duration.ofSeconds(0);
            } else if (clazz == Period.class) {
                instance = Period.ofDays(0);
            } else {
                instance = clazz.getDeclaredConstructor().newInstance();
            }
        } catch (InstantiationException | IllegalAccessException | InvocationTargetException | NoSuchMethodException
                 | ClassNotFoundException e) {
            // Log the error
            System.out.println("erroring");
            System.out.println(e.getMessage());
        }

        Map<String, Object> result = new HashMap<>();
        result.put(variableName, instance);
        return gson.toJson(result);

    }
    private String getParameterType(MethodParameter parameter) {
        if (parameter.hasParameterAnnotation(RequestHeader.class)) {
            return "header";
        } else if (parameter.hasParameterAnnotation(RequestParam.class)) {
            return "query";
        } else if (parameter.hasParameterAnnotation(PathVariable.class)) {
            return "path";
        } else if (parameter.hasParameterAnnotation(RequestBody.class)) {
            return "body";
        }
        return "unknown"; // Default type
    }

    private String getPayloadToPush(){
        Map<Class<?>, List<EndpointInfo>> endpointList = endpointService.getEndpoints();
        ServiceApiDetails serviceApiDetails= new ServiceApiDetails();
        serviceApiDetails.setServiceName(userApplicationName);
        serviceApiDetails.setControllerVsApis(endpointList);
        String json = null;
        try {
            json = new Gson().toJson(serviceApiDetails);
        } catch (Exception e) {
            System.out.println(e.getMessage());
        }

        return json;
    }

    public  void postApiInfoToInventory() throws MalformedURLException {

        HttpURLConnection connection = null;
            try {
                String payload = getPayloadToPush();

                URL apiFlowBackendUrI = URI.create(Constants.apiFlowBackendUrl + Constants.CODE_SCAN_INGESTION_URI).toURL();
                // Logging the start of the API call
                System.out.println("Starting the POST request to the inventory API...");

                URL url = apiFlowBackendUrI;
                connection = (HttpURLConnection) url.openConnection();
                connection.setRequestMethod(HttpRequestMethod.POST.toString());
                connection.setRequestProperty("Content-Type", "application/json");
                connection.setDoOutput(true);
                connection.setRequestProperty("Accept", "application/json");
                connection.setRequestProperty("Connection", "close");
                connection.setRequestProperty(Constants.API_KEY_HEADER,Constants.API_KEY_HEADER_VALUE);
                connection.setRequestProperty(Constants.PARTNER_ID_HEADER,Constants.PARTNER_ID_HEADER_VALUE);

                // Writing the payload
                try (OutputStream os = connection.getOutputStream()) {
                    byte[] input = payload.getBytes("utf-8");
                    os.write(input, 0, input.length);
                }

                int responseCode = connection.getResponseCode();

                // Checking and logging the result
                if (responseCode == HttpURLConnection.HTTP_OK) {
                    System.out.println("Successfully posted to the inventory API.");

                } else {
                    System.out.println("Failed to post to the inventory API. Status code: " + responseCode);
                }
            } catch (Exception e) {
                // Error handling and logging
                System.err.println("An error occurred during the API call to the inventory.");
                e.printStackTrace();
            } finally {
                if (connection != null) {
                    connection.disconnect();
                }
            }
        }





}
