package ai.apiverse.apisuite.mirror.agent;


import ai.apiverse.apisuite.mirror.agent.buffer.Constants;
import ai.apiverse.apisuite.mirror.models.data.APISample;
import ai.apiverse.apisuite.mirror.models.data.AgentConfig;
import ai.apiverse.apisuite.mirror.models.data.ApiOwner;
import ai.apiverse.apisuite.mirror.models.data.ApiSampleWrapper;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;
import com.fasterxml.jackson.databind.module.SimpleModule;
import org.springframework.util.ObjectUtils;
import org.springframework.web.util.UriComponentsBuilder;

import java.io.*;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static java.net.HttpURLConnection.HTTP_OK;

public final class ApimonitorHTTPConnection {
    @SuppressWarnings("CharsetObjectCanBeUsed")
    private static final Charset UTF_8 = Charset.forName("UTF-8");
    final String userApplicationName;
    final String agentId;
    final ApiOwner apiOwner;
    private final Map<String, String> headers;
    private final ApimonitorHttpConnectionConfig httpConnectionConfig;
    private final SDKLogger sdkLogger;
    private final ObjectMapper objectMapper;
    private URL apimonitorSendSampleUrl = null;
    private URL apimonitorConfigUrl = null;

    public ApimonitorHTTPConnection(String apimonitorUrl,
                                    final String userApplicationName,
                                    final String agentId,
                                    ApimonitorHttpConnectionConfig apimonitorHttpConnectionConfig,
                                    SDKLogger sdkLogger,
                                    String authKey,
                                    String environment, ApiOwner apiOwner, Map<String, String> accessHeaders) {
        this.userApplicationName = userApplicationName;
        this.agentId = agentId;
        this.sdkLogger = sdkLogger;
        this.apiOwner = apiOwner;
        headers = new HashMap<>();
        headers.put(SDKVersion.MAJOR_VERSION_KEY, SDKVersion.MAJOR_VERSION);
        headers.put(SDKVersion.MINOR_VERSION_KEY, SDKVersion.MINOR_VERSION);
        headers.put(Constants.API_KEY_HEADER, accessHeaders.get(Constants.API_KEY_HEADER));
        headers.put(Constants.PARTNER_ID_HEADER, accessHeaders.get(Constants.PARTNER_ID_HEADER));
        if (authKey != null) {
            headers.put("authKey", authKey);
        }
        if (environment != null) {
            headers.put("environment", environment);
        }
        this.httpConnectionConfig = apimonitorHttpConnectionConfig;
        this.objectMapper = getObjectMapper();

        try {
            apimonitorSendSampleUrl = URI.create(apimonitorUrl + Constants.SAMPLE_INGESTION_URI).toURL();
        } catch (Exception e) {
            String message = String.format("Unable to create SendSampleUrl from %s. Error: %s", apimonitorUrl, e.getMessage());
            System.out.println(message);
            sdkLogger.forceLog(message);
        }

        try {
            Map<String, String> queryParams = new HashMap<>();
            queryParams.put("appName", userApplicationName);
            queryParams.put("agentId", agentId);
            String configUrl = apimonitorUrl + Constants.CONFIG_URI + getQueryParams(queryParams);
            apimonitorConfigUrl = URI.create(configUrl).toURL();
        } catch (Exception e) {
            String message = String.format("Unable to create ConfigUrl from %s. Error: %s", apimonitorUrl, e.getMessage());
            System.out.println(message);
            sdkLogger.forceLog(message);
        }
    }

    private static String getQueryParams(Map<String, String> params) throws UnsupportedEncodingException {
        if (ObjectUtils.isEmpty(params)) {
            return "";
        }
        UriComponentsBuilder componentsBuilder = UriComponentsBuilder.newInstance();
        for (Map.Entry<String, String> entry : params.entrySet()) {
            componentsBuilder.queryParam(entry.getKey(), URLEncoder.encode(entry.getValue(), StandardCharsets.UTF_8.toString()));
        }

        return componentsBuilder.
                build().
                toString();
    }

    private ObjectMapper getObjectMapper() {
        ObjectMapper objectMapper = new ObjectMapper();
        SimpleModule module = new SimpleModule();

        // Register a simple deserializer for java.net.URI
        module.addDeserializer(URI.class, new StdDeserializer<URI>(URI.class) {
            @Override
            public URI deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
                return URI.create(p.getText());
            }
        });

        objectMapper.registerModule(module);
        return objectMapper;
    }

    public void sendSamples(List<APISample> dataList) {
        try {
            if (null == apimonitorSendSampleUrl) {
                this.sdkLogger.error("SendSampleUrl is null");
                return;
            }
            if (null == dataList || dataList.size() == 0) {
                this.sdkLogger.error("No Samples to send");
                return;
            }
            ApiSampleWrapper apiSampleWrapper = new ApiSampleWrapper();
            apiSampleWrapper.setApiSamples(dataList);
            apiSampleWrapper.setApiOwner(this.apiOwner);
            String payload = null;
            try {
                payload = objectMapper.writeValueAsString(apiSampleWrapper);
            } catch (Exception e) {
                this.sdkLogger.error("Error while parsing content", e);
                return;
            }
            if (null == payload || payload.length() == 0) {
                this.sdkLogger.error("Content is empty");
                return;
            }
            HttpURLConnection connection = (HttpURLConnection) apimonitorSendSampleUrl.openConnection();
            for (Map.Entry<String, String> header : headers.entrySet()) {
                connection.setRequestProperty(header.getKey(), header.getValue());
            }

            connection.setRequestMethod("POST");
            connection.setDoOutput(true);
            connection.setRequestProperty("Content-Type", "application/json");
            connection.setRequestProperty("Accept", "application/json");
            connection.setRequestProperty("Connection", "close");

            connection.setConnectTimeout(httpConnectionConfig.getConnectTimeout());
            connection.setReadTimeout(httpConnectionConfig.getReadTimeout());
            connection.connect();

            try (final OutputStream outputStream = connection.getOutputStream()) {
                try (final BufferedOutputStream bufferedOutputStream = new BufferedOutputStream(outputStream);
                     final Writer writer =
                             new BufferedWriter(new OutputStreamWriter(bufferedOutputStream, UTF_8))) {
                    writer.write(payload);
                    writer.flush();
                }
            } catch (Throwable e) {
                this.sdkLogger.error("An exception occurred while sending Samples", e);
            } finally {
                try {
                    final int responseCode = connection.getResponseCode();
                    if (!isSuccessfulResponseCode(responseCode)) {
                        this.sdkLogger.error(String.format("Send Sample Request failed, API returned %s", responseCode));
                        if (this.sdkLogger.canLogError()) {
                            String errorMessage = getErrorMessageFromStream(connection);
                            this.sdkLogger.error(errorMessage);
                        }
                    }
                } catch (Exception e) {
                    this.sdkLogger.error("Error reading and logging the response stream", e);
                } finally {
                    closeAndDisconnect(connection);
                }
            }
        } catch (Exception e) {
            sdkLogger.error("Error apimonitorHTTPConnection::sendSamples", e);
        }
    }

    public AgentConfig getSDKConfig() {
        try {
            if (null == apimonitorConfigUrl) {
                this.sdkLogger.error("SendSampleUrl is null");
                return null;
            }

            HttpURLConnection connection = (HttpURLConnection) apimonitorConfigUrl.openConnection();
            for (Map.Entry<String, String> header : headers.entrySet()) {
                connection.setRequestProperty(header.getKey(), header.getValue());
            }

            connection.setRequestMethod("GET");
            connection.setDoOutput(false);
            connection.setRequestProperty("Accept", "application/json");
            connection.setRequestProperty("Connection", "close");

            connection.setConnectTimeout(httpConnectionConfig.getConnectTimeout());
            connection.setReadTimeout(httpConnectionConfig.getReadTimeout());
            connection.connect();

            try {
                final int responseCode = connection.getResponseCode();
                if (!isSuccessfulResponseCode(responseCode)) {
                    this.sdkLogger.error(String.format("GetConfig Request failed, API returned %s", responseCode));
                    if (this.sdkLogger.canLogError()) {
                        String errorMessage = getErrorMessageFromStream(connection);
                        this.sdkLogger.error(errorMessage);
                    }
                    return null;
                } else {
                    try (BufferedReader br = new BufferedReader(new InputStreamReader(connection.getInputStream(), UTF_8))) {
                        StringBuilder responseBuilder = new StringBuilder();
                        String responseLine = null;
                        while ((responseLine = br.readLine()) != null) {
                            responseBuilder.append(responseLine.trim());
                        }
                        String response = responseBuilder.toString();
                        if (response.length() == 0) {
                            return null;
                        } else {
                            return objectMapper.readValue(response, new TypeReference<AgentConfig>() {
                            });
                        }
                    }
                }
            } catch (Exception e) {
                this.sdkLogger.error("Error reading and logging the response stream", e);
            } finally {
                closeAndDisconnect(connection);
            }
        } catch (Exception e) {
            sdkLogger.error("Error apimonitorHTTPConnection::getAgentConfig", e);
        }
        return null;
    }

    /**
     * Closes the Response stream and disconnect the connection
     *
     * @param connection the HttpURLConnection
     */
    private void closeAndDisconnect(final HttpURLConnection connection) {
        try {
            connection.getInputStream().close();
        } catch (IOException ignored) {
            // connection is already closed
        } finally {
            connection.disconnect();
        }
    }

    /**
     * Reads the error message from the error stream
     *
     * @param connection the HttpURLConnection
     * @return the error message or null if none
     */
    private String getErrorMessageFromStream(final HttpURLConnection connection) {
        try (final InputStream errorStream = connection.getErrorStream();
             final BufferedReader reader =
                     new BufferedReader(new InputStreamReader(errorStream, UTF_8))) {
            final StringBuilder sb = new StringBuilder();
            String line;
            // ensure we do not add "\n" to the last line
            boolean first = true;
            while ((line = reader.readLine()) != null) {
                if (!first) {
                    sb.append("\n");
                }
                sb.append(line);
                first = false;
            }
            return sb.toString();
        } catch (IOException e) {
            return "Failed to obtain error message while analyzing send failure.";
        }
    }

    /**
     * Returns if response code is OK=200
     *
     * @param responseCode the response code
     * @return true if it is OK=200 or false otherwise
     */
    private boolean isSuccessfulResponseCode(final int responseCode) {
        return responseCode == HTTP_OK;
    }
}
