package ai.apiverse.apisuite.mirror.agent.buffer;

import ai.apiverse.apisuite.mirror.agent.SDKLogger;
import ai.apiverse.apisuite.mirror.models.data.APISample;
import ai.apiverse.apisuite.mirror.models.data.AgentConfig;
import lombok.NonNull;
import org.springframework.scheduling.concurrent.CustomizableThreadFactory;

import java.util.Set;
import java.util.concurrent.*;

public abstract class BufferManagerWorker<K> {
    protected final ConcurrentMap<ApiBufferKey, Buffer> bufferMap = new ConcurrentHashMap<>();
    protected final SDKLogger logger;
    private final ScheduledExecutorService bufferSyncExecutorService = Executors.newSingleThreadScheduledExecutor(new CustomizableThreadFactory("buffer-"));
    private final AgentConfig config;
    ;
    private final String ctUrl;

    public BufferManagerWorker(@NonNull AgentConfig config, String ctUrl, SDKLogger logger) {
        this.config = config;
        this.ctUrl = ctUrl;
        this.logger = logger;
        Runnable runnable = this::syncForKeys;
        bufferSyncExecutorService.scheduleWithFixedDelay(runnable,
                config.getBufferSyncFreqInSec(),
                config.getBufferSyncFreqInSec(),
                TimeUnit.SECONDS);
    }

    public BufferManagerWorker(SDKLogger logger) {
        this.config = null;
        this.ctUrl = null;
        this.logger = logger;
    }

    abstract boolean init();

    abstract boolean offer(K k, APISample apiSample);

    abstract boolean canOffer(K k);

    public AgentConfig getOperatingConfig() {
        return this.config;
    }

    public abstract boolean offer(ApiBufferKey apiBufferKey, APISample apiSample);

    public abstract boolean canOffer(ApiBufferKey apiBufferKey);

    public boolean shutdown() {
        logger.info("Shutting down ApiSampleBufferManagerWorker");
        if (bufferSyncExecutorService.isShutdown()) {
            cleanUpBufferMap();
            return true;
        }

        try {
            bufferSyncExecutorService.shutdown(); // Disable new tasks from being submitted
            if (!bufferSyncExecutorService.awaitTermination(10, TimeUnit.SECONDS)) {
                bufferSyncExecutorService.shutdownNow(); // Cancel currently executing tasks
                if (!bufferSyncExecutorService.awaitTermination(10, TimeUnit.SECONDS)) {
                    logger.error("Still not able to shutdown ApiSampleBufferManagerWorker's bufferSyncExecutorService");
                }
            }
        } catch (InterruptedException e) {
            logger.error("Error while shutting down ApiSampleBufferManagerWorker's bufferSyncExecutorService", e);
        }
        cleanUpBufferMap();
        return bufferSyncExecutorService.isShutdown();
    }

    private void cleanUpBufferMap() {
        syncForKeys();
        for (ApiBufferKey key : this.bufferMap.keySet()) {
            Buffer buffer = this.bufferMap.get(key);
            if (null != buffer) {
                buffer.clear();
            }
        }
        this.bufferMap.clear();
    }

    abstract void syncForKey(ApiBufferKey key);

    private void syncForKeys() {
        Set<ApiBufferKey> keys = this.bufferMap.keySet();
        if (null == keys || keys.size() == 0) {
            return;
        }
        keys.forEach(this::syncForKey);
    }

}
