package ai.apiverse.apisuite.mirror.models.data;

import ai.apiverse.apisuite.mirror.utils.URIUtils;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import lombok.*;
import lombok.experimental.FieldDefaults;

import java.util.List;
import java.util.stream.Collectors;

@Getter
@Setter
@Builder
@NoArgsConstructor
@AllArgsConstructor
@FieldDefaults(level = AccessLevel.PRIVATE)
@JsonIgnoreProperties(ignoreUnknown = true)
@ToString
public class URI {
    String uriPath;
    boolean hasPathVariable;

    public static URI getNonTemplatedURI(String uriPath) {
        return new URI(uriPath, false);
    }

    public static URI getURI(String uriPath) {
        List<String> pathSegments = URIUtils.getPathSegments(uriPath);
        boolean isTemplateURI = false;
        for (String pathSegment : pathSegments) {
            if (URIUtils.isPathSegmentTemplate(pathSegment)) {
                isTemplateURI = true;
                break;
            }
        }
        return new URI(uriPath, isTemplateURI);
    }

    @Override
    public boolean equals(Object object) {
        if (this == object) {
            return true;
        }
        if (object == null) {
            return false;
        }
        // type check and cast
        if (getClass() != object.getClass()) {
            return false;
        }
        URI otherURI = (URI) object;

        if (!isHasPathVariable() && !otherURI.isHasPathVariable()) {
            if (getUriPath().equals(otherURI.getUriPath())) {
                return true;
            }
        }

        List<String> pathSegments = URIUtils.getPathSegments(uriPath);
        List<String> otherURIPathSegments = URIUtils.getPathSegments(otherURI.getUriPath());

        if (pathSegments.size() != otherURIPathSegments.size()) {
            return false;
        }
        for (int idx = 0; idx < pathSegments.size(); idx++) {
            if (!URIUtils.arePathSegmentMatching(pathSegments.get(idx), otherURIPathSegments.get(idx))) {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode() {
        final int PRIME = 59;
        int result = 1;
        result = result * PRIME + URIUtils.getPathSegments(uriPath).size();
        return result;
    }

    public int getSize() {
        return URIUtils.getPathSegments(this.uriPath).size();
    }

    @JsonIgnore
    public List<PathSegment> getPathSegments() {
        return URIUtils.getPathSegments(this.uriPath).stream().map(URIUtils::getPathSegment).collect(Collectors.toList());
    }

    public enum SegmentTemplateDataType {
        NUMBER("{number}"),
        STRING("{string}"),
        UUID("{uuid}"),
        UNKNOWN("{unknown}");

        @Getter
        String pathDisplayName;

        SegmentTemplateDataType(String pathDisplayName) {
            this.pathDisplayName = pathDisplayName;
        }

        public static SegmentTemplateDataType getTemplateDataTypeByDisplayName(String pathDisplayName) {
            for (SegmentTemplateDataType dataType : SegmentTemplateDataType.values()) {
                if (dataType.pathDisplayName.equals(pathDisplayName)) {
                    return dataType;
                }
            }
            return SegmentTemplateDataType.UNKNOWN;
        }
    }

    @Getter
    @Setter
    @NoArgsConstructor
    @FieldDefaults(level = AccessLevel.PRIVATE)
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class URIPathVariable {
        String variableId;
        String variableName;
    }

    @Getter
    @Setter
    @NoArgsConstructor
    @FieldDefaults(level = AccessLevel.PRIVATE)
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class PathSegment {
        private static final String UNKNOWN_DATA_TYPE_SEGMENT_NAME = "{unknown}";
        private static final String INT_DATA_TYPE_SEGMENT_NAME = "{int}";
        private static final String STRING_DATA_TYPE_SEGMENT_NAME = "{string}";
        private static final String UUID_DATA_TYPE_SEGMENT_NAME = "{uuid}";
        String segmentName;
        boolean templatedSegment;
        SegmentTemplateDataType segmentTemplateDataType;

        @Override
        public boolean equals(Object object) {
            if (this == object) {
                return true;
            }
            if (object == null) {
                return false;
            }
            // type check and cast
            if (getClass() != object.getClass()) {
                return false;
            }
            PathSegment otherSegment = (PathSegment) object;
            if (this.isTemplatedSegment() && otherSegment.isTemplatedSegment()) {
                return this.segmentTemplateDataType == otherSegment.segmentTemplateDataType;
            }
            if (this.isTemplatedSegment() || otherSegment.isTemplatedSegment()) {
                return false;
            }
            return this.segmentName.equals(otherSegment.getSegmentName());
        }

        @Override
        public int hashCode() {
            return 79;
        }
    }
}