package ai.apiverse.apisuite.mirror.agent.buffer;

import ai.apiverse.apisuite.mirror.agent.ApimonitorHTTPConnection;
import ai.apiverse.apisuite.mirror.agent.SDKLogger;
import ai.apiverse.apisuite.mirror.agent.config.ConfigManager;
import ai.apiverse.apisuite.mirror.models.data.APISample;

import ai.apiverse.apisuite.mirror.models.data.AgentConfig;
import lombok.NonNull;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Semaphore;

public class RegisteredApiBufferManager extends AbstractBufferManager {

    private final ApimonitorHTTPConnection apimonitorHTTPConnection;
    private final String ctUrl;
    private final SDKLogger logger;
    public RegisteredApiBufferManager(ConfigManager configManager, ApimonitorHTTPConnection apimonitorHTTPConnection, String ctUrl, SDKLogger logger) {
        super(configManager, logger);
        this.apimonitorHTTPConnection = apimonitorHTTPConnection;
        this.ctUrl = ctUrl;
        this.logger = logger;
    }

    BufferManagerWorker<ApiBufferKey> createWorker(AgentConfig newConfig) {
        return new RegisteredApiBufferManagerWorker(newConfig, apimonitorHTTPConnection, logger);
    }

    public class RegisteredApiBufferManagerWorker extends BufferManagerWorker<ApiBufferKey> {

        private final ApimonitorHTTPConnection apimonitorHTTPConnection;
        private final Semaphore semaphore;


        public RegisteredApiBufferManagerWorker(@NonNull AgentConfig config, ApimonitorHTTPConnection apimonitorHTTPConnection, SDKLogger logger) {
            super(config, ctUrl, logger);
            this.apimonitorHTTPConnection = apimonitorHTTPConnection;
            semaphore = new Semaphore(getRegisteredApiCountToCapture());
        }

        @Override
        public boolean init() {
            return true;
        }

        public boolean offer(ApiBufferKey apiBufferKey, APISample apiSample) {
            bufferMap.computeIfAbsent(apiBufferKey, (key) -> new SimpleBuffer(getRegisteredApiBufferSize(apiBufferKey)));
            Buffer buffer = bufferMap.get(apiBufferKey);
            if (buffer != null) {
                return buffer.offer(apiSample);
            } else {
                logger.error("Buffer found null for " + apiBufferKey.getUri());
            }
            return false;
        }

        public boolean canOffer(ApiBufferKey apiBufferKey) {
            if (!getOperatingConfig().getCaptureApiSample()) {
                return false;
            }
            int bufferSize = getRegisteredApiBufferSize(apiBufferKey);
            if (bufferSize == 0) {
                return false;
            }
            Buffer buffer = bufferMap.get(apiBufferKey);
            if (semaphore.tryAcquire()) {
                boolean canOffer = false;
                if (null == buffer) {
                    canOffer = true;
                } else {
                    canOffer = buffer.canOffer();
                }
                semaphore.release();
                return canOffer;
            }
            return false;
        }

        public int getRegisteredApiBufferSize(ApiBufferKey apiBufferKey) {
            AgentConfig agentConfig = getOperatingConfig();
            if (null == agentConfig.getRegisteredApiConfigs() || agentConfig.getRegisteredApiConfigs().size() == 0) {
                return 0;
            }
            for (AgentConfig.ApiConfig apiConfig : agentConfig.getRegisteredApiConfigs()) {
                if (apiConfig.getMethod() == apiBufferKey.getMethod() && apiConfig.getUri().equals(apiBufferKey.getUri())) {
                    return apiConfig.getBufferSize();
                }
            }
            return 0;
        }

        public int getRegisteredApiCountToCapture() {
            AgentConfig agentConfig = getOperatingConfig();
            if (null == agentConfig.getRegisteredApiConfigs() || agentConfig.getRegisteredApiConfigs().size() == 0) {
                return 0;
            }
            int totalApis = 0;
            for (AgentConfig.ApiConfig apiConfig : agentConfig.getRegisteredApiConfigs()) {
                if (null != apiConfig.getBufferSize()) {
                    totalApis += apiConfig.getBufferSize();
                }
            }
            return totalApis;

        }
        void syncForKey(ApiBufferKey key) {
            try {
                Buffer buffer = bufferMap.get(key);
                if (null == buffer) {
                    logger.error("Buffer is null for " + key.getUri());
                    return;
                }
                int iterations = buffer.getContentCount();
                while (iterations -- > 0) {
                    APISample apiSample = buffer.poll();
                    if (null == apiSample) {
                        break;
                    }
                    List<APISample> dataList = new ArrayList<>();
                    dataList.add(apiSample);
                    this.apimonitorHTTPConnection.sendSamples(dataList);
                }
            } catch (Exception exception) {
                logger.error("Error inside syncForKey for key " + key.getUri(), exception);
            }
        }
    }

}
