package ai.apiverse.apisuite.mirror.agent;

import ai.apiverse.apisuite.mirror.agent.buffer.DiscoveredApiBufferManager;
import ai.apiverse.apisuite.mirror.agent.buffer.RegisteredApiBufferManager;
import ai.apiverse.apisuite.mirror.agent.config.ConfigManager;
import ai.apiverse.apisuite.mirror.agent.config.SimpleConfigManager;
import ai.apiverse.apisuite.mirror.models.data.ApiOwner;
import ai.apiverse.apisuite.scanner.ApiScannerConfig;
import ai.apiverse.apisuite.scanner.EndpointService;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.event.ApplicationStartedEvent;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.event.EventListener;
import org.springframework.web.filter.OncePerRequestFilter;

import java.util.List;
import java.util.Random;


@Configuration(proxyBeanMethods = false)
@ConditionalOnProperty(name = "apimonitor.enabled", havingValue = "true")
@Import(ApimonitorHttpConnectionConfig.class)
public class ApimonitorAutoConfiguration implements ApplicationContextAware {
    private ApplicationContext apiPulseContext;
    @Value("${apimonitor.url:}")
    private String url;


    @Value("${apimonitor.applicationName:}")
    private String userApplicationName;

    @Value("${apimonitor.loggingEnabled:false}")
    private boolean loggingEnabled;

    @Value("${apimonitor.logLevel:ERROR}")
    private String logLevel;

    @Value("${apimonitor.authKey:}")
    private String authKey;

    @Value("${apimonitor.environment:}")
    private String environment;

    @Value("${apimonitor.team:}")
    private String team;

    @Value("${apimonitor.maskHeaders:}")
    private List<String> maskHeaders;


    @Bean
    @Qualifier("apimonitor-filter")
    @ConditionalOnProperty(prefix = "apimonitor", name = {"url"})
    OncePerRequestFilter getapimonitorSpringFilter(ApimonitorHttpConnectionConfig apimonitorHttpConnectionConfig) {
        SDKLogger.LogLevel logLevel = null == this.logLevel ? SDKLogger.LogLevel.ERROR : SDKLogger.LogLevel.valueOf(this.logLevel);
        SDKLogger sdkLogger = new SDKLogger(loggingEnabled, logLevel);

        if (null == url || url.length() == 0) {
            String message = "Mandatory Config `apimonitor.url` is missing";
            System.out.println(message);
            sdkLogger.forceLog(message);
            return new ApimonitorNoOpFilter();
        }
        if (null == userApplicationName || userApplicationName.length() == 0) {
            String message = "Mandatory Config `apimonitor.applicationName` is missing";
            System.out.println(message);
            sdkLogger.forceLog(message);
            return new ApimonitorNoOpFilter();
        }
        url = url.trim();
        userApplicationName = userApplicationName.trim();

        Random random = new Random();
        int randomId = random.nextInt(1000);
        String agentId = String.valueOf(randomId);
        ApiOwner apiOwner = new ApiOwner();
        apiOwner.setEnv(environment);
        apiOwner.setTeam(team);
        apiOwner.setServiceName(userApplicationName);

        ApimonitorHTTPConnection apimonitorHTTPConnection = new ApimonitorHTTPConnection(url, userApplicationName, agentId, apimonitorHttpConnectionConfig, sdkLogger, authKey, environment,apiOwner);

        ConfigManager configManager = new SimpleConfigManager(url, userApplicationName, agentId, apimonitorHTTPConnection, sdkLogger);
        configManager.init();

        RegisteredApiBufferManager registeredApiBufferManager = new RegisteredApiBufferManager(configManager, apimonitorHTTPConnection, url, sdkLogger);
        registeredApiBufferManager.init();

        DiscoveredApiBufferManager discoveredApiBufferManager = new DiscoveredApiBufferManager(configManager, apimonitorHTTPConnection, url, sdkLogger);
        discoveredApiBufferManager.init();

        ApimonitorApiProcessor apimonitorApiProcessor = new ApimonitorApiProcessor(registeredApiBufferManager, discoveredApiBufferManager, maskHeaders, sdkLogger);
        ApimonitorSpringFilter apimonitorSpringFilter = new ApimonitorSpringFilter(configManager, apimonitorApiProcessor, userApplicationName, sdkLogger);
        apimonitorSpringFilter.init();
        String message = String.format("apimonitor Initialised!\napplicationName:: %s\nurl:: %s\nagentId:: %s\nSDK Version:: %s.%s\nHeader mask keys:: %s",
                userApplicationName, url, agentId, SDKVersion.MAJOR_VERSION, SDKVersion.MINOR_VERSION, String.join(", ", maskHeaders));
        System.out.println(message);
        sdkLogger.forceLog(message);
        return apimonitorSpringFilter;
    }

    @EventListener(ApplicationStartedEvent.class)
    public void initscan()  {
        try {
            EndpointService endpointService = new EndpointService();
            endpointService.setApplicationContext(this.apiPulseContext);
            ApiScannerConfig apiScannerConfig = ApiScannerConfig.getInstance();
            apiScannerConfig.setEndpointService(endpointService);
            apiScannerConfig.setLogEnable(loggingEnabled);
            apiScannerConfig.printLog("maintaining api inventory","");
            apiScannerConfig.setApiFlowBackendUrl(url);
            apiScannerConfig.setUserApplicationName(userApplicationName);
            apiScannerConfig.postApiInfoToInventory();
        }catch (Exception e){
            ApiScannerConfig apiScannerConfig = ApiScannerConfig.getInstance();
            apiScannerConfig.printLog("exception occured while trying to maintain api inventory",e.getMessage());
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.apiPulseContext=applicationContext;
    }
}
