package ai.apiverse.apisuite.mirror.agent.buffer;

import ai.apiverse.apisuite.mirror.agent.ApimonitorHTTPConnection;
import ai.apiverse.apisuite.mirror.agent.SDKLogger;
import ai.apiverse.apisuite.mirror.agent.config.ConfigManager;
import ai.apiverse.apisuite.mirror.models.data.APISample;
import ai.apiverse.apisuite.mirror.models.data.AgentConfig;
import lombok.NonNull;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Semaphore;
public class DiscoveredApiBufferManager extends AbstractBufferManager {
    private final ApimonitorHTTPConnection apimonitorHTTPConnection;
    private final String ctUrl;
    private final SDKLogger logger;
    public DiscoveredApiBufferManager(ConfigManager configManager, ApimonitorHTTPConnection apimonitorHTTPConnection, String ctUrl, SDKLogger logger) {
        super(configManager, logger);
        this.apimonitorHTTPConnection = apimonitorHTTPConnection;
        this.ctUrl = ctUrl;
        this.logger = logger;
    }

    BufferManagerWorker<ApiBufferKey> createWorker(AgentConfig newConfig) {
        return new DiscoveredApiBufferManagerWorker(newConfig, apimonitorHTTPConnection, logger);
    }

    public class DiscoveredApiBufferManagerWorker extends BufferManagerWorker<ApiBufferKey> {

        private final ApimonitorHTTPConnection apimonitorHTTPConnection;
        private final Semaphore semaphore;

        public DiscoveredApiBufferManagerWorker(@NonNull AgentConfig config, ApimonitorHTTPConnection apimonitorHTTPConnection, SDKLogger logger) {
            super(config, ctUrl, logger);
            semaphore = new Semaphore(config.getDiscoveryBufferSize());
            this.apimonitorHTTPConnection = apimonitorHTTPConnection;
        }

        @Override
        public boolean init() {
            return true;
        }

        public boolean offer(ApiBufferKey apiBufferKey, APISample apiSample) {
            bufferMap.computeIfAbsent(apiBufferKey, (key) -> new SimpleBuffer(getOperatingConfig().getDiscoveryBufferSizePerApi()));
            Buffer buffer = bufferMap.get(apiBufferKey);
            if (buffer != null) {
                return buffer.offer(apiSample);
            } else {
                logger.error("Buffer found null for " + apiBufferKey.getUri());
            }
            return false;
        }

        public boolean canOffer(ApiBufferKey apiBufferKey) {
            if (semaphore.tryAcquire()) {
                boolean canOffer = false;
                if (getOperatingConfig().getCaptureApiSample()) {
                    Buffer buffer = bufferMap.get(apiBufferKey);
                    if (null != buffer) {
                        canOffer = buffer.canOffer();
                    } else {
                        canOffer = bufferMap.size() < getOperatingConfig().getDiscoveryBufferSize();
                    }
                }
                semaphore.release();
                return canOffer;
            }
            return false;
        }
        void syncForKey(ApiBufferKey key) {
            try {
                Buffer buffer = bufferMap.get(key);
                if (null == buffer) {
                    logger.error("Buffer is null for " + key.getUri());
                    return;
                }
                int iterations = buffer.getContentCount();
                if (iterations == 0) {
                    bufferMap.remove(key);
                    return;
                }
                List<APISample> contents = new ArrayList<>();
                while (iterations -- > 0) {
                    APISample apiSample = buffer.poll();
                    if (null == apiSample) {
                        bufferMap.remove(key);
                        break;
                    }
                    contents.add(apiSample);
                }
                if (contents.size() == 0) {
                    return;
                }
                this.apimonitorHTTPConnection.sendSamples(contents);
            } catch (Exception exception) {
                logger.error("Error inside syncForKey for key " + key.getUri(), exception);
            }
        }
    }
}
