package ai.benshi.android.sdk.e_commerce.builders

import ai.benshi.android.sdk.core.BenshiLog
import ai.benshi.android.sdk.core.event_types.EventType
import ai.benshi.android.sdk.core.utils.CoreConstants
import ai.benshi.android.sdk.e_commerce.event_models.CheckoutObject
import ai.benshi.android.sdk.e_commerce.event_models.OrderItem
import ai.benshi.android.sdk.e_commerce.event_types.CurrencyCode
import ai.benshi.android.sdk.e_commerce.utils.ECommerceConstants
import android.content.Context
import com.google.gson.Gson
import java.lang.IllegalArgumentException

/**
 * Created by Moiz Hassan on 25, January,2022
 */

class BsLogCheckoutEvent {

    /**
     * BsLogCheckoutEvent is required to log checkout event for orders. You can trigger this event
     * when the order has been placed or unable to place. you need to provide orderId in both cases.
     * If you don't have the orderId in case of not being successful then you can pass cartId for
     * that order.
     */

    data class Builder(
        private var context: Context? = null,
        private var order_id: String? = null,
        private var is_successful: Boolean? = null,
        private var price: Float? = null,
        private var currency: String? = null,
        private var tax: Float? = null,
        private var cart_id: String? = "",
        private var orderList: ArrayList<OrderItem> = arrayListOf(),
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * init is required to pass context to the SDK method and log the event.
         * if not provided, it will throw an exception on runtime.
         */
        fun init(context: Context) = apply { this.context = context }

        /**
         * setOrderId is required to log the orderId for the order being successful or failed
         * to place. In case of not having any orderId for failed cases you can pass the cartId
         * or a corresponding value that can be used to track in the catalog about the order
         * placement details.
         */
        fun setOrderId(order_id: String) = apply { this.order_id = order_id }

        /**
         * isSuccessful is required to log the successful placement of the order. False in case of
         * order placement is not successful.
         */
        fun isSuccessful(is_successful: Boolean) = apply { this.is_successful = is_successful }

        /**
         * setPrice is required to log the total price of the order being logged. Price format
         * should be in accordance to the currency selected.
         */
        fun setPrice(price: Float) = apply { this.price = price }

        /**
         * setCurrency is required to log the currency for for the order logged. Currency should
         * be in ISO 4217 format. For ease, SDK provides the enums to log the currency so that it
         * would be easy to log. You can also use the string function to provide the currency.
         * Below is the function for the logging currency using enum CurrencyCode.
         */
        fun setCurrency(currency: CurrencyCode) = apply { this.currency = currency.name }

        /**
         * setCurrency is required to log the currency for for the order logged. Currency should
         * be in ISO 4217 format. For Ease, SDK provides the enums to log the currency so that it
         * would be easy to log. You can also use the string function to provide the currency.
         * Below is the function for the logging currency using String. Remember to use the same
         * strings as provided in the enums or else the event will be discarded.
         */
        fun setCurrency(currency: String) = apply {
            if(CoreConstants.enumContains<CurrencyCode>(currency)){
                this.currency = currency
            }else{
                throw IllegalArgumentException("Invalid currency_code provided")
            }
        }

        /**
         * setTax is required to log the total tax for the order being logged. Tax format
         * should be in accordance to the currency selected.
         */
        fun setTax(tax: Float) = apply { this.tax = tax }

        /**
         * setCartId is required to log the cart Id for the order being checked out. If you don't
         * have specific cart Id, You can pass the userId instead or can pass null.
         */
        fun setCartId(cart_id: String?) = apply { this.cart_id = cart_id }

        /**
         * addOrder can be used to add the item being orders in to the checkout list. This log can
         * be used to add one item to the log at a time. Order item should be in a valid format.
         * With elements of the orderObject as:
         * OrderItem(itemID, price, quantity, promoId) Promo Id can be an empty string or no
         * value at all if the item does not have a promo offer that is obtained by the user.
         * You can add multiple addOrder functions to one checkout event to include all the
         * items in the order.
         */
        fun addOrder(orderItem: OrderItem) = apply {
            this.orderList.add(orderItem)
        }

        /**
         * addOrderList can be used to add the whole list to the log at once, the format should be
         * ArrayList<OrderItem> to log the event successfully. Order item should be in a valid format.
         * With elements of the orderObject as:
         * OrderItem(itemID, price, quantity, promoId) Promo Id can be an empty string or no
         * value at all if the item does not have a promo offer that is obtained by the user.
         * You should use only one addOrderList with checkout event or else the list will only
         * save the later one.
         */
        fun addOrderList(orderList: ArrayList<OrderItem>) = apply {
            this.orderList.addAll(orderList)
        }

        /**
         * addOrderList can be used to add the whole list to the log at once, the format should be
         * ArrayList<OrderItem> to log the event successfully. But the input param is of type
         * string , this is special use case for react native logs where list can be passed as
         * Json string and can be used to log the event. Order item should be in a valid format.
         * With elements of the orderObject as:
         * OrderItem(itemID, price, quantity, promoId) Promo Id can be an empty string or no
         * value at all if the item does not have a promo offer that is obtained by the user.
         * You should use only one addOrderList with checkout event or else the list will only
         * save the later one.
         */
        fun addOrderList(orderListString: String) = apply {
            val orderItems: Array<OrderItem> = Gson().fromJson(
                orderListString,
                Array<OrderItem>::class.java
            )
            val orderList = java.util.ArrayList(orderItems.toMutableList())
            this.orderList.addAll(orderList)
        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if context provided is null or not
                 * provided at all.
                 */
                context == null -> {
                    throw NullPointerException("init is required to provide context.")
                }

                /**
                 * Will throw and exception if the orderId provided is null or no value is
                 * provided at all.
                 */
                order_id == null -> {
                    throw NullPointerException("orderId is required.")
                }

                /**
                 * Will throw and exception if the is_successful provided is null or no value is
                 * provided at all.
                 */
                is_successful == null -> {
                    throw NullPointerException("is_successful is required.")
                }

                /**
                 * Will throw and exception if the price provided is null or no value is
                 * provided at all.
                 */
                price == null -> {
                    throw NullPointerException("price is required.")
                }

                /**
                 * Will throw and exception if the currency provided is null or no value is
                 * provided at all.
                 */
                currency == null -> {
                    throw NullPointerException("currency is required.")
                }

                /**
                 * Will throw and exception if the tax provided is null or no value is
                 * provided at all.
                 */
                tax == null -> {
                    throw NullPointerException("tax is required.")
                }

                /**
                 * Will throw and exception if the orderList provided is null or no value is
                 * provided at all for the order Items.
                 */
                orderList.isNullOrEmpty() -> {
                    throw NullPointerException("order Items are required.")
                }
                else -> {

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val checkoutObject = CheckoutObject(order_id!!, is_successful!!, price,
                        currency, tax, cart_id, orderList, meta)
                    BenshiLog.track(
                        context!!, ECommerceConstants.moduleName, EventType.checkout,
                        checkoutObject, update_immediately
                    )
                }
            }
        }
    }
}