package ai.benshi.android.sdk.e_commerce.builders

import ai.benshi.android.sdk.core.BenshiLog
import ai.benshi.android.sdk.core.event_types.EventType
import ai.benshi.android.sdk.core.utils.CoreConstants
import ai.benshi.android.sdk.e_commerce.event_models.ItemObject
import ai.benshi.android.sdk.e_commerce.event_types.CurrencyCode
import ai.benshi.android.sdk.e_commerce.event_types.ItemType
import ai.benshi.android.sdk.e_commerce.utils.ECommerceConstants
import android.content.Context
import java.lang.IllegalArgumentException

/**
 * Created by Moiz Hassan on 24, January,2022
 */

class BsLogItemEvent {

    /**
     * BsLogItemEvent is required to log item related events which included when an item is viewed
     * and when an item's detail is viewed.
     */

    data class Builder(
        private var context: Context? = null,
        private var item_type: String? = null,
        private var item_id: String? = null,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * init is required to pass context to the SDK method and log the event.
         * if not provided, it will throw an exception on runtime.
         */
        fun init(context: Context) = apply { this.context = context }

        /**
         * setType is required to set Type for the type of log in this case, if a user is viewing
         * an item or it's details. setType log is used to define if the log is about the item
         * itself or if the user id going into the details. you can provide the values using 2
         * different methods, one is for enum based and the other is for string based.
         * Below is the method for the enum based approach.
         */
        fun setType(item_type: ItemType) = apply { this.item_type = item_type.name }

        /**
         * setType is required to set Type for the type of log in this case, if a user is viewing
         * an item or it's details. setType log is used to define if the log is about the item
         * itself or if the user id going into the details. you can provide the values using 2
         * different methods, one is for enum based and the other is for string based.
         * Below is the method for the string based approach. Remember to note that for string
         * input you need to use the same names as defined in the enum or else the events will
         * be discarded.
         */
        fun setType(item_type: String) = apply {
            if(CoreConstants.enumContains<ItemType>(item_type)){
                this.item_type = item_type
            }else{
                throw IllegalArgumentException("Invalid item_type provided")
            }
        }

        /**
         * setItemId is required to log the Id for item the events are being logged. Details
         * about the item are to be provided in the catalog for more details about the item.
         */
        fun setItemId(item_id: String) = apply { this.item_id = item_id }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if context provided is null or not
                 * provided at all.
                 */
                context == null -> {
                    throw NullPointerException("init is required to provide context.")
                }

                /**
                 * Will throw and exception if the type provided is null or no value is
                 * provided at all.
                 */
                item_type == null -> {
                    throw NullPointerException("type is required.")
                }

                /**
                 * Will throw and exception if the itemId provided is null or no value is
                 * provided at all.
                 */
                item_id == null -> {
                    throw NullPointerException("item_id is required.")
                }
                else -> {

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val itemObject = ItemObject(item_type!!, item_id!!, meta)
                    BenshiLog.track(
                        context!!, ECommerceConstants.moduleName, EventType.item,
                        itemObject, update_immediately
                    )
                }
            }
        }
    }
}