package ai.benshi.android.sdk.e_commerce.builders

import ai.benshi.android.sdk.core.BenshiLog
import ai.benshi.android.sdk.core.event_types.EventType
import ai.benshi.android.sdk.core.utils.CoreConstants
import ai.benshi.android.sdk.e_commerce.event_models.CheckoutObject
import ai.benshi.android.sdk.e_commerce.event_models.ItemObject
import ai.benshi.android.sdk.e_commerce.event_models.ListObject
import ai.benshi.android.sdk.e_commerce.event_types.ItemType
import ai.benshi.android.sdk.e_commerce.event_types.ListAction
import ai.benshi.android.sdk.e_commerce.event_types.ListType
import ai.benshi.android.sdk.e_commerce.utils.ECommerceConstants
import android.content.Context
import java.lang.IllegalArgumentException

/**
 * Created by Moiz Hassan on 25, January,2022
 */

class BsLogListEvent {

    /**
     * BsLogListEvent is used to log the list regarding events. This log provides support for
     * different types of lists in the ecommerce section that includes cart, favorites, reminders,
     * orders and a more general option as other. Remember to note to that if you use 'other then '
     * include the list type in meta to that to be added in the log.
     */

    data class Builder(
        private var context: Context? = null,
        private var list_type: String? = null,
        private var list_id: String? = null,
        private var item_id: String? = null,
        private var list_action: String? = null,
        private var quantity: Int? = null,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * init is required to pass context to the SDK method and log the event.
         * if not provided, it will throw an exception on runtime.
         */
        fun init(context: Context) = apply { this.context = context }

        /**
         * setListType is required to pass the type of list the logged is triggered on. By default
         * the SDK provides 4 main list types for e-commerce apps. Which includes cart, favourite,
         * reminder and order list types. If the list you want to log is not covered by the
         * provided then you can use the 'other' list type that can be used to track custom list
         * types. Remember to note that with 'other' list Type, pass the list type name in the meta
         * of the log so that it can be included.
         * setListType provides 2 approaches for logging list events, one is with enums and the
         * other is with string. Below is the function to log listType event using enum type.
         */
        fun setListType(list_type: ListType) = apply { this.list_type = list_type.name }

        /**
         * setListType is required to pass the type of list the logged is triggered on. By default
         * the SDK provides 4 main list types for e-commerce apps. Which includes cart, favourite,
         * reminder and order list types. If the list you want to log is not covered by the
         * provided then you can use the 'other' list type that can be used to track custom list
         * types. Remember to note that with 'other' list Type, pass the list type name in the meta
         * of the log so that it can be included.
         * setListType provides 2 approaches for logging list events, one is with enums and the
         * other is with string. Below is the function to log listType event using string type.
         * Remember to note that with string type, you need to pass the values as provided
         * in the enum or else the events will be discarded
         */
        fun setListType(list_type: String) = apply {
            if(CoreConstants.enumContains<ListType>(list_type)){
                this.list_type = list_type
            }else{
                throw IllegalArgumentException("Invalid list_type provided")
            }
        }

        /**
         * setListId can be used to log the listId the event is logged for. For use cases where
         * there is no listId, you can skip this or pass null. SDK will auto associate the itemId
         * as the list ID. It is recommended to include the listIds for the defined list types so
         * that they can be tracked.
         */
        fun setListId(list_id: String?) = apply { this.list_id = list_id }

        /**
         * setItemId is required to log the Id for the item/product the which is used to log with
         * the actions being used. this Item should be of the product being added, removed or
         * being edited.
         */
        fun setItemId(item_id: String?) = apply { this.item_id = item_id }

        /**
         * setListAction is required to pass the actions for list the logged is triggered on. By
         * default the SDK provides 5 main list actions for e-commerce apps. Which includes
         * addItem, removeItem, editItem, view and discard list actions.
         * setListAction provides 2 approaches for logging list events, one is with enums and the
         * other is with string. Below is the function to log listAction event using enum type.
         */
        fun setListAction(list_action: ListAction) = apply { this.list_action = list_action.name }

        /**
         * setListAction is required to pass the actions for list the logged is triggered on. By
         * default the SDK provides 5 main list actions for e-commerce apps. Which includes
         * addItem, removeItem, editItem, view and discard list actions.
         * setListAction provides 2 approaches for logging list events, one is with enums and the
         * other is with string. Below is the function to log listAction event using string type.
         * Remember to note that with string type, you need to pass the values as provided
         * in the enum or else the events will be discarded
         */
        fun setListAction(list_action: String) = apply { this.list_action = list_action }

        /**
         * setQuantity is used to log the quantity of the item that is added, removed or edited.
         * You can skip this or pass null as well for actions like view or discard list as SDK
         * will override the quantity in such use cases.
         */
        fun setQuantity(quantity: Int) = apply { this.quantity = quantity }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if context provided is null or not
                 * provided at all.
                 */
                context == null -> {
                    throw NullPointerException("init is required to provide context.")
                }

                /**
                 * Will throw and exception if the listType provided is null or no value is
                 * provided at all.
                 */
                list_type == null -> {
                    throw NullPointerException("listType is required.")
                }

                /**
                 * Will throw and exception if the itemId provided is null or no value is
                 * provided for use cases when the listAction is not view or discard.
                 */
                item_id == null -> {
                    if(list_action != ListAction.view.name &&
                        list_action != ListAction.discard.name)
                    throw NullPointerException("itemId is required.")
                }

                /**
                 * Will throw and exception if the listAction provided is null or no value is
                 * provided at all.
                 */
                list_action == null -> {
                    throw NullPointerException("listAction is required.")
                }
                else -> {

                    /**
                     * Will throw and exception if the quantity is not provided or null in case
                     * the listType is not favorite and reminder and action is not view or discard.
                     */
                    if( list_type != ListType.favourite.name &&
                        list_type != ListType.reminder.name &&
                        /**
                         * ListType with values as favourite or reminder, you don't need to provide
                         * quantity. In such cases you can omit the setQuantity and easily
                         * trigger view, add, remove, edit and discard events.
                         */
                        list_action != ListAction.view.name &&
                        list_action != ListAction.discard.name &&
                        /**
                         * ListType with values as favourite or reminder, you don't need to provide
                         * quantity. In such cases you can omit the setQuantity.
                         */
                        quantity == null){
                        throw NullPointerException("quantity is required for add_item, remove_item " +
                                "and edit_item action events")
                    }else{

                        /**
                         * setListId is an optional value keeping the use case that most apps
                         * don't have a backend supported listId for favourite or reminders and
                         * in some cases for cart as well. For such use cases, you can either
                         * leave the setListId or can also pass null in that. In case of NULL,
                         * SDK will pass the itemID as the listId. It is recommended to use the
                         * ListId for such lists to support the catalog as well.
                         */
                        if(list_id == null && item_id != null){
                            list_id = item_id
                        }
                        if(quantity == null){
                            quantity = 0
                        }

                        /**
                         * Parsing the values into an object and passing to the setup block to queue
                         * the event based on its priority.
                         */

                        val listObject = ListObject(list_type!!, list_id!!, item_id!!,
                            list_action!!, quantity!!, meta)
                        BenshiLog.track(
                            context!!, ECommerceConstants.moduleName, EventType.list,
                            listObject, update_immediately
                        )
                    }
                }
            }
        }
    }
}