package ai.benshi.android.sdk.e_commerce.builders

import ai.benshi.android.sdk.core.BenshiLog
import ai.benshi.android.sdk.core.event_types.EventType
import ai.benshi.android.sdk.core.utils.CoreConstants
import ai.benshi.android.sdk.e_commerce.event_models.ItemObject
import ai.benshi.android.sdk.e_commerce.event_models.RateObject
import ai.benshi.android.sdk.e_commerce.event_types.ItemType
import ai.benshi.android.sdk.e_commerce.utils.ECommerceConstants
import android.content.Context

/**
 * Created by Moiz Hassan on 24, January,2022
 */

class BsLogRateEvent {

    /**
     * BsLogRateEvent is required to log user experience for a particular order.
     * In order to log this event, developer needs to provide the orderId for which the rating
     * is being obtained and the rating itself.
     */

    data class Builder(
        private var context: Context? = null,
        private var order_id: String? = null,
        private var rate: Float? = null,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * init is required to pass context to the SDK method and log the event.
         * if not provided, it will throw an exception on runtime.
         */
        fun init(context: Context) = apply { this.context = context }

        /**
         * setRating is required to log teh rating provided by the user for a particular order.
         * the value should be a float value which represents the numerical value for stars
         * provided.
         */
        fun setRating(rate: Float) = apply { this.rate = rate }

        /**
         * setOrderId is required to associate the rating obtained for the order. OrderId should
         * be a valid orderId and can be tracked from the catalog
         */
        fun setOrderId(order_id: String) = apply { this.order_id = order_id }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if context provided is null or not
                 * provided at all.
                 */
                context == null -> {
                    throw NullPointerException("init is required to provide context.")
                }

                /**
                 * Will throw and exception if the rate provided is null or no value is
                 * provided at all.
                 */
                rate == null -> {
                    throw NullPointerException("rate value is required.")
                }

                /**
                 * Will throw and exception if the orderId provided is null or no value is
                 * provided at all.
                 */
                order_id == null -> {
                    throw NullPointerException("orderId is required.")
                }

                else -> {

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val rateObject = RateObject(order_id!!, rate!!, meta)
                    BenshiLog.track(
                        context!!, ECommerceConstants.moduleName, EventType.rate,
                        rateObject, update_immediately
                    )
                }
            }
        }
    }
}