package ai.benshi.android.sdk.e_commerce.builders

import ai.benshi.android.sdk.core.BenshiLog
import ai.benshi.android.sdk.core.builders.Benshi
import ai.benshi.android.sdk.core.utils.CoreConstants
import ai.benshi.android.sdk.e_commerce.currency_exchange.CurrencyAPIHandler
import ai.benshi.android.sdk.e_commerce.currency_exchange.EComLog
import ai.benshi.android.sdk.e_commerce.event_models.ItemModel
import ai.benshi.android.sdk.e_commerce.event_models.ViewItemObject
import ai.benshi.android.sdk.e_commerce.event_types.CurrencyCode
import ai.benshi.android.sdk.e_commerce.event_types.EComEventType
import ai.benshi.android.sdk.e_commerce.event_types.ItemStockStatus
import ai.benshi.android.sdk.e_commerce.event_types.ItemViewAction
import ai.benshi.android.sdk.e_commerce.utils.ECommerceConstants
import android.content.Context
import com.google.gson.Gson
import kotlin.math.roundToInt

/**
 * Created by Moiz Hassan on 24, January,2022
 */

class BsLogItemEvent {

    /**
     * BsLogItemEvent is required to log item related events which included when an item is viewed
     * and when an item's detail is viewed.
     */

    data class Builder(
        private var context: Context? = null,
        private var item_action: String? = null,
        private var item: ItemModel? = null,
        private var search_id: String? = null,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * init is required to pass context to the SDK method and log the event.
         * if not provided, it will throw an exception on runtime.
         */
        fun init(context: Context) = apply { this.context = context }

        /**
         * setItemViewAction is required to set Type for the type of log in this case, if a user is viewing
         * an item or it's details. setType log is used to define if the log is about the item
         * itself or if the user id going into the details. you can provide the values using 2
         * different methods, one is for enum based and the other is for string based.
         * Below is the method for the enum based approach.
         */
        fun setItemViewAction(item_viewAction: ItemViewAction) = apply { this.item_action = item_viewAction.name }

        /**
         * setItemViewAction is required to set Type for the type of log in this case, if a user is viewing
         * an item or it's details. setType log is used to define if the log is about the item
         * itself or if the user id going into the details. you can provide the values using 2
         * different methods, one is for enum based and the other is for string based.
         * Below is the method for the string based approach. Remember to note that for string
         * input you need to use the same names as defined in the enum or else the events will
         * be discarded.
         */
        fun setItemViewAction(item_viewAction: String) = apply {
            if (CoreConstants.enumContains<ItemViewAction>(item_viewAction)) {
                this.item_action = item_viewAction
            } else {
                throw IllegalArgumentException("Invalid Item View Action provided")
            }
        }

        /**
         * setItemId is required to log the Id for item the events are being logged. Details
         * about the item are to be provided in the catalog for more details about the item.
         */
        fun setItemId(item_id: String) = apply {
            this.item?.id = item_id
            item?.quantity = 1 //default set to 1 for item actions
        }
        /**
         * setItemPrice is required to log the price for item the events are being logged. Details
         * about the item are to be provided in the catalog for more details about the item.
         */
        fun setItemPrice(item_price: Float) = apply {
            this.item?.price = ((item_price * 100.0).roundToInt() / 100.0).toFloat()
        }
        fun setItemPrice(item_price: Int) = apply { this.item?.price = item_price.toFloat() }
        fun setItemPrice(item_price: Double) = apply {
            this.item?.price = ((item_price * 100.0).roundToInt() / 100.0).toFloat() }

        /**
         * setItemCurrency is required to log the currency for item the events are being logged. Details
         * about the item are to be provided in the catalog for more details about the item.
         */
        fun setItemCurrency(currency_code: CurrencyCode) = apply { this.item?.currency = currency_code.name }
        fun setItemCurrency(currency_code: String) = apply {
            if (CoreConstants.enumContains<CurrencyCode>(currency_code)) {
                this.item?.currency = currency_code
            } else {
                throw IllegalArgumentException("Invalid Currency Code provided")
            }
        }

        /**
         * setItemStockStatus is required to log the stock status for item the events are being
         * logged. Details about the item are to be provided in the catalog for more details about
         * the item.
         */
        fun setItemStockStatus(stock_status: ItemStockStatus) = apply { this.item?.stockStatus = stock_status.name }
        fun setItemStockStatus(stock_status: String) = apply {
            if (CoreConstants.enumContains<ItemStockStatus>(stock_status)) {
                this.item?.stockStatus = stock_status
            } else {
                throw IllegalArgumentException("Invalid Stock Status provided")
            }
        }

        /**
         * setItemPromoId is required to log the promo Id for item the events are being
         * logged. Details about the item are to be provided in the catalog for more details about
         * the item. If there is not promo associated with the product, you do not need to pass it.
         */
        fun setItemPromoId(promo_id: String) = apply { this.item?.promoId = promo_id }

        /**
         * setItem can be used to pass the whole item as an object as well. You can use the POJO
         * ItemModel to parse the data int he required format and pass that to this function to
         * log the event.
         */
        fun setItem(item: ItemModel) = apply {
            item.quantity = 1 //default set to 1 for item actions
            this.item = item
        }

        /**
         * setItem can be used to pass the whole item as a Json String object as well. You can use
         * the POJO ItemModel to parse the data int he required format and pass that to this
         * function as a string to log the event. You can use Gson to convert the object to string
         * but SDK will parse the Json string back to POJO so pass it in the log. This method
         * should be used with caution and is suitable for react native bridge.
         */
        fun setItem(itemJsonString: String) = apply {
            val item: ItemModel = Gson().fromJson(itemJsonString, ItemModel::class.java)
            item.quantity = 1 //default set to 1 for item actions
            this.item = item
        }

        /**
         * setSearchId is used to associate the search id with the item being viewed by the user.
         * It is required to track if the item is a result of some search performed by the user in
         * the app.
         */
        fun setSearchId(search_id: String?) = apply { this.search_id = search_id }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if context provided is null or not
                 * provided at all.
                 */
                context == null -> {
                    throw NullPointerException("init is required to provide context.")
                }

                /**
                 * Will throw and exception if the type provided is null or no value is
                 * provided at all.
                 */
                item_action == null -> {
                    throw NullPointerException("Item Action is required.")
                }

                /**
                 * Will throw and exception if the itemId provided is null or no value is
                 * provided at all.
                 */
                item?.id == null -> {
                    throw NullPointerException("Item Id is required.")
                }
                item?.price == null -> {
                    throw NullPointerException("Item Price is required.")
                }
                item?.currency == null -> {
                    throw NullPointerException("Item Currency is required.")
                }
//                item?.stockStatus == null -> {
//                    throw NullPointerException("Item Stock Status is required.")
//                }
//                item?.promoId == null -> {
//                    throw NullPointerException("Stem Promo Id is required.")
//                }

                else -> {

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val itemObject = ViewItemObject(item_action!!, item!!, search_id, null, meta)
                    if(item!!.currency == CurrencyCode.USD.name){
                        itemObject.usd_rate = 1f
                        BenshiLog.track(
                            context!!, ECommerceConstants.contentBlockName, EComEventType.item.name,
                            itemObject, update_immediately
                        )
                    }else{
                        CurrencyAPIHandler().initInterface()
                        EComLog.track(
                            context!!, item!!.currency, EComEventType.item.name,
                            itemObject, update_immediately
                        )
                    }

                }
            }
        }
    }
}