package ai.benshi.android.sdk.e_commerce.utils.impression_listener.ui

import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView

/**
 * Wraps around a [RowVisibilityCalculator] and applies the given
 * [VisibilityThreshold] to the list of currently visible rows, so that only
 * those which meet the threshold are reported back upstream.
 */
internal class ConstrainedRowVisibilityCalculator(
    recyclerView: RecyclerView,
    layoutManager: LinearLayoutManager,
    private val clock: Long,
    private val threshold: VisibilityThreshold
) {
    private val rawVisibilityCalculator = RowVisibilityCalculator(
        clock, recyclerView, layoutManager
    )

    fun calculateVisibleRows(): List<RowVisibility> {
        // Raw visibilities must be calculated *before* the current time is taken, because
        // when raw visibilities are calculated for the first time, they will also be reading the
        // current time in order to keep track of the timestamp when they first became visible.
        // Therefore, if we read the current time prior to making this call, there will be a
        // negative difference between "now" and when the rows became visible.
        val rawVisibilities = rawVisibilityCalculator.calculateRowVisibilities()

        val now = System.currentTimeMillis()

        return rawVisibilities
            .filter { visibility ->
                val timeVisible = now - visibility.visibleSinceMillis
                visibility.percentageVisible >= threshold.percentageThreshold
                        && timeVisible >= threshold.timeThresholdMillis
            }
    }
}
