package ai.benshi.android.sdk.e_commerce.impression_listener.trackers

import ai.benshi.android.sdk.core.BenshiLog
import ai.benshi.android.sdk.core.catalog_models.CatalogSubject
import ai.benshi.android.sdk.core.utils.CoreConstants
import ai.benshi.android.sdk.core.utils.ExceptionManager
import ai.benshi.android.sdk.e_commerce.builders.BsLogItemEvent
import ai.benshi.android.sdk.e_commerce.catalog_models.DrugModel
import ai.benshi.android.sdk.e_commerce.catalog_models.internal_catalog_models.InternalDrugModel
import ai.benshi.android.sdk.e_commerce.event_models.RecyclerItemModel
import ai.benshi.android.sdk.e_commerce.event_types.ItemViewAction
import ai.benshi.android.sdk.e_commerce.utils.ECommerceConstants
import ai.benshi.android.sdk.e_commerce.impression_listener.calculation.AsyncCollectionDiffCalculator
import android.app.Activity
import kotlinx.coroutines.Dispatchers

/**
 * Allows you to track impressions based on a collection of [RecyclerItemModel] (i.e. from a
 * RecyclerView or some other collection-based view).
 *
 * This class should be retained as a singleton to ensure the collection view keys are properly
 * tracked across their lifecycle.
 */
internal class TrackCollectionsUseCase {
    private val collectionDiffers =
        mutableMapOf<String, AsyncCollectionDiffCalculator<RecyclerItemModel>>()

    /**
     * To be called when a collection view first becomes visible with its initial content. Only
     * for semantic clarity. Same as calling [onCollectionUpdated]
     */
    fun onCollectionVisible(
        sourceActivity: Activity?,
        collectionViewKey: String,
        searchId: String,
        visibleModel: List<RecyclerItemModel>
    ) = onCollectionUpdated(sourceActivity, collectionViewKey, searchId, visibleModel)

    /**
     * To be called when the collection view with the given [collectionViewKey] has been entirely
     * removed/hidden from the viewport. This is so that the end-time of impressions that started
     * via [onCollectionUpdated] can be tracked.
     *
     * Only for semantic clarity. Same as calling [onCollectionUpdated] with an empty list.
     */
    fun onCollectionHidden(
        sourceActivity: Activity?,
        searchId: String,
        collectionViewKey: String
    ) =
        onCollectionUpdated(sourceActivity, collectionViewKey, searchId, emptyList())

    /**
     * To be called when the collection view with the given [collectionViewKey] has a
     * new list of *currently* visible content.
     *
     * Impressions will be tracked for every new piece of content. Content that existed in the
     * previous call to this function will not result in a new impression being logged.
     *
     * Content that was visible the last time this function was called, but is no longer visible,
     * will be considered "dropped"--and this will be tracked as the end of its total impression
     * time (start time = when it first shows up in [onCollectionUpdated], end time when
     * [onCollectionUpdated] is called again and the content is no longer present in the
     * [visibleModel] list provided.
     *
     * Note: The [visibleModel] parameter is not the full list of content backing the collection
     * view; rather, it should be a list representing the content/rows currently within the
     * viewport.
     */
    fun onCollectionUpdated(
        sourceActivity: Activity?,
        collectionViewKey: String,
        searchId: String,
        visibleModel: List<RecyclerItemModel>
    ) {

        if (visibleModel.isEmpty()) onNoContent(collectionViewKey)

        val differ = collectionDiffers.getOrPut(collectionViewKey) {
            AsyncCollectionDiffCalculator(
                calculationContext = Dispatchers.Default,
                notificationContext = Dispatchers.Main
            )
        }

        differ.scheduleDiffCalculation(
            newBaseline = visibleModel,
            onResult = { newDiff ->
                onNewDiff(
                    searchId,
                    originalImpressionTime = System.currentTimeMillis(),
                    result = newDiff
                )
            }
        )
    }

    private fun onNoContent(collectionViewKey: String) {
        // If we begin logging end impressions, they would need to be handled from both the callback
        // passed to the differ, and also here
        collectionDiffers.remove(collectionViewKey)
    }

    private fun onNewDiff(
        searchId: String,
        originalImpressionTime: Long,
        result: AsyncCollectionDiffCalculator.DiffResult<RecyclerItemModel>
    ) {
        result.newItems.forEach { newContent ->
            onStartImpression(
                searchId,
                time = originalImpressionTime,
                model = newContent
            )
        }
    }

    private fun onStartImpression(
        searchId: String,
        time: Long,
        model: RecyclerItemModel
    ) {

        if (!CoreConstants.impressionItemsList.contains(model.itemProperties.id)) {
            CoreConstants.impressionItemsList.add(model.itemProperties.id)
            BsLogItemEvent.Builder().init(ECommerceConstants.context!!)
                .setItem(model.itemProperties).setItemViewAction(ItemViewAction.impression)
                .setSearchId(searchId).build()

            callCatalogAPI(model.itemProperties.id, model.drugProperties)

        }
    }

    private fun callCatalogAPI(itemId : String, drugModel: DrugModel) {
        when {
            drugModel.supplier_id.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException("supplier_id")
            }
            drugModel.supplier_name.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException("supplier_name")
            }
            drugModel.active_ingredients.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException("active_ingredients")
            }
            drugModel.market_id.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException("market_id")
            }
            else -> {
                val activeIngredients = drugModel.active_ingredients!!.joinToString(separator = "|·|")

                val internalDrugModel = InternalDrugModel(
                    id = itemId,
                    name = checkIfNull(drugModel.name),
                    market_id = checkIfNull(drugModel.market_id),
                    description = checkIfNull(drugModel.description),
                    supplier_id = checkIfNull(drugModel.supplier_id),
                    supplier_name = checkIfNull(drugModel.supplier_name),
                    producer = checkIfNull(drugModel.producer),
                    packaging = checkIfNull(drugModel.packaging),
                    active_ingredients = checkIfNull(activeIngredients),
                    drug_form = checkIfNull(drugModel.drug_form),
                    drug_strength = checkIfNull(drugModel.drug_strength),
                    atc_anatomical_group =  checkIfNull(drugModel.ATC_anatomical_group),
                    otc_or_ethical = checkIfNull(drugModel.OTC_or_ethical))
                BenshiLog().updateCatalogItem(CatalogSubject.drug, internalDrugModel)

            }
        }
    }

    private fun checkIfNull(inputValue: String?): String {
        if (inputValue.isNullOrEmpty()) {
            return ""
        }
        return inputValue
    }
}
