package ai.benshi.android.sdk.e_commerce.impression_listener.trackers

import ai.benshi.android.sdk.core.utils.CoreConstants
import ai.benshi.android.sdk.e_commerce.event_models.RecyclerItemModel
import ai.benshi.android.sdk.e_commerce.impression_listener.ui.ImpressionThreshold
import ai.benshi.android.sdk.e_commerce.impression_listener.ui.Tracker
import android.app.Activity
import android.content.Context
import android.content.ContextWrapper
import androidx.recyclerview.widget.RecyclerView

/**
 * Given a [RecyclerView], allows you to track impressions of [RecyclerItemModel] based on which
 * rows are currently visible to the user. Will update impressions as the user scrolls.
 *
 * This class should be retained as a singleton to ensure the RecyclerView view keys are properly
 * tracked across their lifecycle.
 */
internal class TrackRecyclerViewUseCase(
    private val coreImpressionsUseCase: TrackCollectionsUseCase
) {
    private val currentRecyclerViews = mutableMapOf<String, Tracker<RecyclerItemModel>>()

    fun trackRecyclerView(
        recyclerView: RecyclerView,
        currentDataProvider: () -> List<RecyclerItemModel>,
        searchId: String,
        impressionThresholdBlock: (ImpressionThreshold.Builder.() -> Unit)?
    ) {
        val threshold = if (impressionThresholdBlock != null)
            ImpressionThreshold.Builder().apply(impressionThresholdBlock).build()
        else ImpressionThreshold.Builder().build()

        trackRecyclerView(recyclerView, currentDataProvider, searchId, threshold)
    }

    fun trackRecyclerView(
        recyclerView: RecyclerView,
        currentDataProvider: () -> List<RecyclerItemModel>,
        searchId: String,
        impressionThreshold: ImpressionThreshold
    ) {
        val rvKey = "RV-${recyclerView.id}"
        CoreConstants.impressionItemsList.clear()
        // Cancel & remove any existing trackers for this RV
        stopAndRemoveTracking(rvKey)

        // Re-add a tracker (begins tracking automatically)
        currentRecyclerViews[rvKey] = Tracker(
            clock = System.currentTimeMillis(),
            recyclerView = recyclerView,
            visibilityThreshold = impressionThreshold,
            currentDataProvider = currentDataProvider,
            onVisibleRowsChanged = { onVisibleRowsChanged(recyclerView, rvKey, searchId, it) },
            onRecyclerViewDetached = { onRVDetached(recyclerView, rvKey, searchId) }
        )
    }

    private fun onVisibleRowsChanged(
        recyclerView: RecyclerView,
        rvKey: String,
        searchId: String,
        data: List<RecyclerItemModel>
    ) {
        coreImpressionsUseCase.onCollectionUpdated(
            recyclerView.getActivity(),
            rvKey,
            searchId,
            data
        )
    }

    private fun onRVDetached(recyclerView: RecyclerView, rvKey: String, searchId: String) {
        stopAndRemoveTracking(rvKey)
        coreImpressionsUseCase.onCollectionHidden(recyclerView.getActivity(), searchId, rvKey)
    }

    private fun stopAndRemoveTracking(recyclerViewKey: String) =
        currentRecyclerViews.remove(recyclerViewKey)?.stopTracking()

    private fun RecyclerView.getActivity(): Activity? {
        var context: Context = context
        while (context is ContextWrapper) {
            if (context is Activity) {
                return context
            }
            context = context.baseContext
        }
        return null
    }
}
