package ai.benshi.android.sdk.e_commerce.builders

import ai.benshi.android.sdk.core.BenshiLog
import ai.benshi.android.sdk.core.catalog_models.CatalogSubject
import ai.benshi.android.sdk.core.utils.CoreConstants
import ai.benshi.android.sdk.core.utils.ExceptionManager
import ai.benshi.android.sdk.e_commerce.catalog_models.DrugCatalogModel
import ai.benshi.android.sdk.e_commerce.catalog_models.internal_catalog_models.InternalDrugModel
import ai.benshi.android.sdk.e_commerce.event_models.InternalCurrencyCode
import ai.benshi.android.sdk.e_commerce.event_models.ItemModel
import ai.benshi.android.sdk.e_commerce.event_models.ViewItemObject
import ai.benshi.android.sdk.e_commerce.event_types.EComEventType
import ai.benshi.android.sdk.e_commerce.event_types.ItemStockStatus
import ai.benshi.android.sdk.e_commerce.event_types.ItemType
import ai.benshi.android.sdk.e_commerce.event_types.ItemViewAction
import ai.benshi.android.sdk.e_commerce.utils.ECommerceConstants
import android.content.Context
import com.google.gson.Gson
import kotlin.math.roundToInt

/**
 * Created by Moiz Hassan on 24, January,2022
 */

class BsLogItemEvent {

    /**
     * BsLogItemEvent is required to log item related events which included when an item is viewed
     * and when an item's detail is viewed.
     */

    data class Builder(
        private var context: Context? = null,
        var item_action: String? = null,
        var item_value: ItemModel? = null,
        private var item_id: String? = null,
        private var item_quantity: Int = 1, // default set to 1 for item actions
        private var item_price: Float? = null,
        private var item_currency: String? = null,
        private var item_type: String? = null,
        private var item_stockStatus: String? = null,
        private var item_promoId: String? = null,
        var search_id: String? = null,
        var drug_model: DrugCatalogModel? = null,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {
        private lateinit var itemObject : ViewItemObject

        /**
         * init is required to pass context to the SDK method and log the event.
         * if not provided, it will throw an exception on runtime.
         */
        fun init(context: Context) = apply { this.context = context }

        /**
         * setItemViewAction is required to set Type for the type of log in this case, if a user is viewing
         * an item or it's details. setType log is used to define if the log is about the item
         * itself or if the user id going into the details. you can provide the values using 2
         * different methods, one is for enum based and the other is for string based.
         * Below is the method for the enum based approach.
         */
        fun setItemViewAction(item_viewAction: ItemViewAction) = apply { this.item_action = item_viewAction.name }

        /**
         * setItemViewAction is required to set Type for the type of log in this case, if a user is viewing
         * an item or it's details. setType log is used to define if the log is about the item
         * itself or if the user id going into the details. you can provide the values using 2
         * different methods, one is for enum based and the other is for string based.
         * Below is the method for the string based approach. Remember to note that for string
         * input you need to use the same names as defined in the enum or else the events will
         * be discarded.
         */
        fun setItemViewAction(item_viewAction: String) = apply {
            if (CoreConstants.enumContains<ItemViewAction>(item_viewAction)) {
                this.item_action = item_viewAction
            } else {
                ExceptionManager.throwEnumException(ItemViewAction::class.java.simpleName)
            }
        }

        /**
         * setItemId is required to log the Id for item the events are being logged. Details
         * about the item are to be provided in the catalog for more details about the item.
         */
        fun setItemId(item_id: String) = apply {
            this.item_id = item_id
        }

        /**
         * setItemQuantity is required to log the quantity for item the events are being logged. Details
         * about the item are to be provided in the catalog for more details about the item.
         */
        fun setItemQuantity(item_quantity: Int) = apply {
            this.item_quantity = item_quantity }

        /**
         * setItemPrice is required to log the price for item the events are being logged. Details
         * about the item are to be provided in the catalog for more details about the item.
         */
        fun setItemPrice(item_price: Float) = apply {
            this.item_price = ((item_price * 100.0).roundToInt() / 100.0).toFloat() }
        fun setItemPrice(item_price: Int) = apply { this.item_price = item_price.toFloat() }
        fun setItemPrice(item_price: Double) = apply {
            this.item_price = ((item_price * 100.0).roundToInt() / 100.0).toFloat() }

        /**
         * setItemCurrency is required to log the currency for item the events are being logged. Details
         * about the item are to be provided in the catalog for more details about the item.
         */
//        fun setItemCurrency(currency_code: CurrencyCode) = apply { this.item_currency = currency_code.name }
        fun setItemCurrency(currency_code: String) = apply {
            if (CoreConstants.enumContains<InternalCurrencyCode>(currency_code)) {
                this.item_currency = currency_code
            } else {
                ExceptionManager.throwEnumException("CurrencyCode")
            }
        }

        /**
         * setItemType is required to log the type for item the events are being
         * logged. Details about the item are to be provided in the catalog for more details about
         * the item.
         */
        fun setItemType(item_type: ItemType) = apply { this.item_type = item_type.name }
        fun setItemType(item_type: String) = apply {
            if (CoreConstants.enumContains<ItemType>(item_type)) {
                this.item_type = item_type
            } else {
                ExceptionManager.throwEnumException(ItemType::class.java.simpleName)
            }
        }

        /**
         * setItemStockStatus is required to log the stock status for item the events are being
         * logged. Details about the item are to be provided in the catalog for more details about
         * the item.
         */
        fun setItemStockStatus(stock_status: ItemStockStatus) = apply { this.item_stockStatus = stock_status.name }
        fun setItemStockStatus(stock_status: String) = apply {
            if (CoreConstants.enumContains<ItemStockStatus>(stock_status)) {
                this.item_stockStatus = stock_status
            } else {
                ExceptionManager.throwEnumException(ItemStockStatus::class.java.simpleName)
            }
        }

        /**
         * setItemPromoId is required to log the promo Id for item the events are being
         * logged. Details about the item are to be provided in the catalog for more details about
         * the item. If there is not promo associated with the product, you do not need to pass it.
         */
        fun setItemPromoId(promo_id: String) = apply { this.item_promoId = promo_id }

        /**
         * setItem can be used to pass the whole item as an object as well. You can use the POJO
         * ItemModel to parse the data int he required format and pass that to this function to
         * log the event.
         */
        fun setItem(item: ItemModel) = apply {
            this.item_value = item
        }

        /**
         * setItem can be used to pass the whole item as a Json String object as well. You can use
         * the POJO ItemModel to parse the data int he required format and pass that to this
         * function as a string to log the event. You can use Gson to convert the object to string
         * but SDK will parse the Json string back to POJO so pass it in the log. This method
         * should be used with caution and is suitable for react native bridge.
         */
        fun setItem(itemJsonString: String) = apply {
            val item: ItemModel = Gson().fromJson(itemJsonString, ItemModel::class.java)
            this.item_value = item
        }

        /**
         * setSearchId is used to associate the search id with the item being viewed by the user.
         * It is required to track if the item is a result of some search performed by the user in
         * the app.
         */
        fun setSearchId(search_id: String?) = apply { this.search_id = search_id }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        fun setDrugProperties(drugProperties: DrugCatalogModel) =
            apply { this.drug_model = drugProperties }

        fun setDrugProperties(drugProperties: String?) =
            apply { if (drugProperties != null) {
                val drugProperties: DrugCatalogModel = Gson().fromJson(drugProperties, DrugCatalogModel::class.java)
                this.drug_model = drugProperties
            }
        }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if context provided is null or not
                 * provided at all.
                 */
                context == null -> {
                    ExceptionManager.throwInitException()
                }

                /**
                 * Will throw and exception if the type provided is null or no value is
                 * provided at all.
                 */
                item_action == null -> {
                    ExceptionManager.throwIsRequiredException(ItemViewAction::class.java.simpleName)
                }

                else -> {

                    if(item_value == null){
                        /**
                         * Will throw and exception if the itemId provided is null or no value is
                         * provided at all.
                         */
                        when {
                            item_id == null -> {
                                ExceptionManager.throwIsRequiredException("item_id")
                            }
                            item_price == null -> {
                                ExceptionManager.throwIsRequiredException("item_price")
                            }
                            item_currency == null -> {
                                ExceptionManager.throwIsRequiredException("item_currency")
                            }
                            item_quantity < 0 -> {
                                ExceptionManager.throwItemQuantityException()
                            }
                            item_type == null -> {
                                ExceptionManager.throwIsRequiredException("item_type")
                            }
//                            item_stockStatus == null -> {
//                                ExceptionManager.throwIsRequiredException("item_stockStatus")
//                            }
//                            item_promoId == null -> {
//                                ExceptionManager.throwIsRequiredException("item_promoId")
//                            }
                            else -> {
                                item_value = ItemModel(item_id!!, item_quantity, item_price!!, item_currency!!,
                                    item_type!!, item_stockStatus, item_promoId)
                            }
                        }
                    }

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */

                    itemObject = ViewItemObject(item_action!!, item_value!!, search_id, null, meta)

                    if (item_action == ItemViewAction.view.name && this.drug_model != null) {
                        callCatalogAPI()
                    }

                    if (item_value!!.currency == InternalCurrencyCode.USD.name) {
                        itemObject.usd_rate = 1f
                        BenshiLog().track(
                            context!!, ECommerceConstants.contentBlockName, EComEventType.item.name,
                            itemObject, update_immediately
                        )
                    } else {
                        BenshiLog().getUSDRate(context!!, item_value!!.currency, ::getUSDRateAndLogEvent)
                    }
                }
            }
        }

        private fun getUSDRateAndLogEvent(usdRate : Float){
            itemObject.usd_rate = usdRate
            BenshiLog().track(
                context!!, ECommerceConstants.contentBlockName, EComEventType.item.name,
                itemObject, update_immediately
            )
        }

        private fun callCatalogAPI() {
            when {
                drug_model!!.supplier_id.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("supplier_id")
                }
                drug_model!!.supplier_name.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("supplier_name")
                }
                drug_model!!.active_ingredients.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("active_ingredients")
                }
                drug_model!!.market_id.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("market_id")
                }
                else -> {
                    val activeIngredients = this.drug_model!!.active_ingredients!!.joinToString(separator = "|·|")

                    val internalDrugModel = InternalDrugModel(
                        id = item_value!!.id,
                        name = checkIfNull(this.drug_model!!.name),
                        market_id = checkIfNull(this.drug_model!!.market_id),
                        description = checkIfNull(this.drug_model!!.description),
                        supplier_id = checkIfNull(this.drug_model!!.supplier_id),
                        supplier_name = checkIfNull(this.drug_model!!.supplier_name),
                        producer = checkIfNull(this.drug_model!!.producer),
                        packaging = checkIfNull(this.drug_model!!.packaging),
                        active_ingredients = checkIfNull(activeIngredients),
                        drug_form = checkIfNull(this.drug_model!!.drug_form),
                        drug_strength = checkIfNull(this.drug_model!!.drug_strength),
                        atc_anatomical_group =  checkIfNull(this.drug_model!!.ATC_anatomical_group),
                        otc_or_ethical = checkIfNull(this.drug_model!!.OTC_or_ethical))
                    BenshiLog().updateCatalogItem(CatalogSubject.drug, internalDrugModel)
                }
            }
        }

        private fun checkIfNull(inputValue: String?): String {
            if (inputValue.isNullOrEmpty()) {
                return ""
            }
            return inputValue
        }
    }
}
