package ai.benshi.android.sdk.e_commerce.builders

import ai.benshi.android.sdk.core.BenshiLog
import ai.benshi.android.sdk.core.utils.CoreConstants
import ai.benshi.android.sdk.core.utils.ExceptionManager
import ai.benshi.android.sdk.e_commerce.event_models.CancelOrderObject
import ai.benshi.android.sdk.e_commerce.event_models.CartObject
import ai.benshi.android.sdk.e_commerce.event_models.InternalCurrencyCode
import ai.benshi.android.sdk.e_commerce.event_models.ItemModel
import ai.benshi.android.sdk.e_commerce.event_types.*
import ai.benshi.android.sdk.e_commerce.utils.ECommerceConstants
import android.content.Context
import com.google.gson.Gson
import kotlin.math.roundToInt

/**
 * Created by Moiz Hassan on 19, September, 2022
 */

class BsLogCancelOrderEvent {

    /**
     * BsLogCancelOrderEvent is used to log the event when the order is canceled.
     */

    data class Builder(
        private var context: Context? = null,
        internal var order_id: String? = null,
        internal var item_types: ArrayList<String> = arrayListOf(),
        internal var reason: String = "",
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        private lateinit var cartObject : CartObject

        /**
         * init is required to pass context to the SDK method and log the event.
         * if not provided, it will throw an exception on runtime.
         */
        fun init(context: Context) = apply { this.context = context }

        /**
         * setOrderId can be used to log the orderId the event is logged for. if the order ID is
         * not there then it is recommended to include the unique cartId for the cart items so
         * that they can be tracked. In case the cart Id is also not there, you can use the userId
         * as a unique element for the order.
         */
        fun setOrderId(order_id: String) = apply { this.order_id = order_id }

        /**
         * setOrderItemTypes is required to provide the list of item types that are present
         * in the order being cancelled
         */
        fun addOrderItemTypesList(item_types: ArrayList<String>) = apply {
            this.item_types.clear()
            for(item in item_types){
                if (CoreConstants.enumContains<ItemType>(item)) {
                    this.item_types.add(item)
                } else {
                    ExceptionManager.throwEnumException(ItemType::class.java.simpleName)
                }
            }
        }

        fun addOrderItemType(item_type: ItemType) = apply {
            this.item_types.add(item_type.name)
        }
        fun addOrderItemType(item_type: String) = apply {
            if (CoreConstants.enumContains<ItemType>(item_type)) {
                this.item_types.add(item_type)
            } else {
                ExceptionManager.throwEnumException(ItemType::class.java.simpleName)
            }
        }


        /**
         * setCancelReason is required to define the reason for which the item is being cancelled.
         */
        fun setCancelReason(reason: String) = apply {
            this.reason = reason
        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if context provided is null or not
                 * provided at all.
                 */
                context == null -> {
                    ExceptionManager.throwInitException()
                }

                /**
                 * Will throw an exception for the developer if order_id provided is null or not
                 * provided at all.
                 */
                order_id.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("order_id")
                }

                /**
                 * Will throw an exception for the developer if item_types provided is null or not
                 * provided at all.
                 */
                item_types.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("item_types")
                }

//                /**
//                 * Will throw an exception for the developer if reason provided is null or not
//                 * provided at all.
//                 */
//                reason.isNullOrEmpty() -> {
//                    ExceptionManager.throwIsRequiredException("reason")
//                }

                else -> {
                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val cancelOrderObject = CancelOrderObject(order_id!!, item_types,
                        reason, meta)
                    BenshiLog().track(
                        context!!, ECommerceConstants.contentBlockName, EComEventType.cancel_order.name,
                        cancelOrderObject, update_immediately
                    )
                }
            }
        }

    }
}
