package ai.benshi.android.sdk.e_commerce.builders

import ai.benshi.android.sdk.core.BenshiLog
import ai.benshi.android.sdk.core.utils.CoreConstants
import ai.benshi.android.sdk.core.utils.ExceptionManager
import ai.benshi.android.sdk.e_commerce.event_models.CartObject
import ai.benshi.android.sdk.e_commerce.event_models.InternalCurrencyCode
import ai.benshi.android.sdk.e_commerce.event_models.ItemModel
import ai.benshi.android.sdk.e_commerce.event_types.*
import ai.benshi.android.sdk.e_commerce.utils.ECommerceConstants
import android.content.Context
import com.google.gson.Gson
import kotlin.math.roundToInt

/**
 * Created by Moiz Hassan on 25, January,2022
 */

class BsLogCartEvent {

    /**
     * BsLogCartEvent is used to log the events related to cart. You can use this event to log
     * when an item was added or removed form the cart.
     */

    data class Builder(
        private var context: Context? = null,
        var cart_id: String? = null,
        var cart_action: String? = null,
        var item_value: ItemModel? = null,
        private var item_id: String? = null,
        private var item_quantity: Int = 1, // default set to 1 for item actions
        private var item_price: Float? = null,
        private var item_currency: String? = null,
        private var item_type: String? = null,
        private var item_stockStatus: String? = null,
        private var item_promoId: String? = null,
        var cart_price: Float? = null,
        var currency_value: String? = null,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        private lateinit var cartObject : CartObject

        /**
         * init is required to pass context to the SDK method and log the event.
         * if not provided, it will throw an exception on runtime.
         */
        fun init(context: Context) = apply { this.context = context }

        /**
         * setCartId can be used to log the cartId the event is logged for. It is recommended to
         * include the unique cartId for the cart items so that they can be tracked.
         */
        fun setCartId(cart_id: String) = apply { this.cart_id = cart_id }

        /**
         * setCartAction is required to pass the actions for cart the logged is triggered on. By
         * default the SDK provides 2 main list actions for e-commerce apps. Which includes
         * addItem and removeItem
         * setCartAction provides 2 approaches for logging list events, one is with enums and the
         * other is with string. Below is the function to log cartAction event using enum type.
         */
        fun setCartAction(cart_action: CartAction) = apply { this.cart_action = cart_action.name }

        /**
         * setCartAction is required to pass the actions for cart the logged is triggered on. By
         * default the SDK provides 2 main list actions for e-commerce apps. Which includes
         * addItem and removeItem.
         * setCartAction provides 2 approaches for logging list events, one is with enums and the
         * other is with string. Below is the function to log listAction event using string type.
         * Remember to note that with string type, you need to pass the values as provided
         * in the enum or else the events will be discarded
         */
        fun setCartAction(cart_action: String) = apply {
            if (CoreConstants.enumContains<CartAction>(cart_action)) {
                this.cart_action = cart_action
            } else {
                ExceptionManager.throwEnumException(CartAction::class.java.simpleName)
            }
        }

        /**
         * setItemId is required to log the Id for item the events are being logged. Details
         * about the item are to be provided in the catalog for more details about the item.
         */
        fun setItemId(item_id: String) = apply {
            this.item_id = item_id
        }

        /**
         * setItemQuantity is required to log the quantity for item the events are being logged. Details
         * about the item are to be provided in the catalog for more details about the item.
         */
        fun setItemQuantity(item_quantity: Int) = apply {
            this.item_quantity = item_quantity }

        /**
         * setItemPrice is required to log the price for item the events are being logged. Details
         * about the item are to be provided in the catalog for more details about the item.
         */
        fun setItemPrice(item_price: Float) = apply {
            this.item_price = ((item_price * 100.0).roundToInt() / 100.0).toFloat() }
        fun setItemPrice(item_price: Int) = apply { this.item_price = item_price.toFloat() }
        fun setItemPrice(item_price: Double) = apply {
            this.item_price = ((item_price * 100.0).roundToInt() / 100.0).toFloat() }

        /**
         * setItemCurrency is required to log the currency for item the events are being logged. Details
         * about the item are to be provided in the catalog for more details about the item.
         */
//        fun setItemCurrency(currency_code: CurrencyCode) = apply { this.item_currency = currency_code.name }
        fun setItemCurrency(currency_code: String) = apply {
            if (CoreConstants.enumContains<InternalCurrencyCode>(currency_code)) {
                this.item_currency = currency_code
            } else {
                ExceptionManager.throwEnumException("CurrencyCode")
            }
        }

        /**
         * setItemType is required to log the type for item the events are being
         * logged. Details about the item are to be provided in the catalog for more details about
         * the item.
         */
        fun setItemType(item_type: ItemType) = apply { this.item_type = item_type.name }
        fun setItemType(item_type: String) = apply {
            if (CoreConstants.enumContains<ItemType>(item_type)) {
                this.item_type = item_type
            } else {
                ExceptionManager.throwEnumException(ItemType::class.java.simpleName)
            }
        }

        /**
         * setItemStockStatus is required to log the stock status for item the events are being
         * logged. Details about the item are to be provided in the catalog for more details about
         * the item.
         */
        fun setItemStockStatus(stock_status: ItemStockStatus) = apply { this.item_stockStatus = stock_status.name }
        fun setItemStockStatus(stock_status: String) = apply {
            if (CoreConstants.enumContains<ItemStockStatus>(stock_status)) {
                this.item_stockStatus = stock_status
            } else {
                ExceptionManager.throwEnumException(ItemStockStatus::class.java.simpleName)
            }
        }

        /**
         * setItemPromoId is required to log the promo Id for item the events are being
         * logged. Details about the item are to be provided in the catalog for more details about
         * the item. If there is not promo associated with the product, you do not need to pass it.
         */
        fun setItemPromoId(promo_id: String) = apply { this.item_promoId = promo_id }

        /**
         * setItem can be used to pass the whole item as an object as well. You can use the POJO
         * ItemModel to parse the data int he required format and pass that to this function to
         * log the event.
         */
        fun setItem(item: ItemModel) = apply {
            this.item_value = item
        }

        /**
         * setItem can be used to pass the whole item as a Json String object as well. You can use
         * the POJO ItemModel to parse the data int he required format and pass that to this
         * function as a string to log the event. You can use Gson to convert the object to string
         * but SDK will parse the Json string back to POJO so pass it in the log. This method
         * should be used with caution and is suitable for react native bridge.
         */
        fun setItem(itemJsonString: String) = apply {
            val item: ItemModel = Gson().fromJson(itemJsonString, ItemModel::class.java)
            this.item_value = item
        }

        /**
         * setCartPrice is required to log the total price for cart being logged. Details
         * about the cart are to be provided in the catalog.
         */
        fun setCartPrice(cart_price: Int) = apply { this.cart_price = cart_price.toFloat() }
        fun setCartPrice(cart_price: Float) = apply {
            this.cart_price = ((cart_price * 100.0).roundToInt() / 100.0).toFloat() }
        fun setCartPrice(cart_price: Double) = apply {
            this.cart_price = ((cart_price * 100.0).roundToInt() / 100.0).toFloat() }

        /**
         * setCurrency is required to log the currency for cart price being logged. Details
         * about the cart are to be provided in the catalog.
         */
//        fun setCurrency(currency_code: CurrencyCode) = apply { this.currency_value = currency_code.name }
        fun setCurrency(currency_code: String) = apply {
            if (CoreConstants.enumContains<InternalCurrencyCode>(currency_code)) {
                this.currency_value = currency_code
            } else {
                ExceptionManager.throwEnumException("CurrencyCode")
            }
        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if context provided is null or not
                 * provided at all.
                 */
                context == null -> {
                    ExceptionManager.throwInitException()
                }

                /**
                 * Will throw an exception for the developer if cart_id provided is null or not
                 * provided at all.
                 */
                cart_id.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("cart_id")
                }

                /**
                 * Will throw an exception for the developer if cart_action provided is null or not
                 * provided at all.
                 */
                cart_action.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(CartAction::class.java.simpleName)
                }

                /**
                 * Will throw an exception for the developer if cart_price provided is null or not
                 * provided at all.
                 */
                cart_price == null -> {
                    ExceptionManager.throwIsRequiredException("cart_price")
                }

                /**
                 * Will throw an exception for the developer if currency provided is null or not
                 * provided at all.
                 */
                currency_value.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(InternalCurrencyCode::class.java.simpleName)
                }

                else -> {

                    if(item_value == null){
                        /**
                         * Will throw and exception if the itemId provided is null or no value is
                         * provided at all.
                         */
                        when {
                            item_id.isNullOrEmpty() -> {
                                ExceptionManager.throwIsRequiredException("item_id")
                            }
                            item_price == null -> {
                                ExceptionManager.throwIsRequiredException("item_price")
                            }
                            item_currency.isNullOrEmpty() -> {
                                ExceptionManager.throwIsRequiredException("item_currency")
                            }
                            item_quantity < 0 -> {
                                ExceptionManager.throwItemQuantityException()
                            }
                            item_type.isNullOrEmpty() -> {
                                ExceptionManager.throwIsRequiredException("item_type")
                            }

                            //                        else if(item_stockStatus == null) {
                            //                            throw NullPointerException("Item Stock Status is required.")
                            //                        }
                            //                        else if(item_promoId == null) {
                            //                            throw NullPointerException("Stem Promo Id is required.")
                            //                        }
                            else -> {
                                item_value = ItemModel(item_id!!, item_quantity, item_price!!, item_currency!!,
                                    item_type!!, item_stockStatus, item_promoId)
                            }
                        }
                    }else{
                        ECommerceConstants.isItemValueObjectValid(item_value!!)
                    }

                    if (currency_value != item_value!!.currency) {
                        ExceptionManager.throwCurrencyNotSameException("cart")
                    }

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    cartObject = CartObject(cart_id!!, cart_action!!,
                        item_value!!, cart_price!!, currency_value!!, null, meta)
                    if (currency_value == InternalCurrencyCode.USD.name) {
                        cartObject.usd_rate = 1f
                        BenshiLog().track(
                            context!!, ECommerceConstants.contentBlockName, EComEventType.cart.name,
                            cartObject, update_immediately
                        )
                    } else {
                        BenshiLog().getUSDRate(context!!, item_value!!.currency, ::getUSDRateAndLogEvent)
                    }
                }
            }
        }

        private fun getUSDRateAndLogEvent(usdRate : Float){
            cartObject.usd_rate = usdRate
            BenshiLog().track(
                context!!, ECommerceConstants.contentBlockName, EComEventType.cart.name,
                cartObject, update_immediately
            )
        }

    }
}
