package ai.benshi.android.sdk.e_commerce.builders

import ai.benshi.android.sdk.core.BenshiLog
import ai.benshi.android.sdk.core.utils.CoreConstants
import ai.benshi.android.sdk.core.utils.ExceptionManager
import ai.benshi.android.sdk.e_commerce.event_models.CheckoutObject
import ai.benshi.android.sdk.e_commerce.event_models.InternalCurrencyCode
import ai.benshi.android.sdk.e_commerce.event_models.ItemModel
import ai.benshi.android.sdk.e_commerce.event_types.EComEventType
import ai.benshi.android.sdk.e_commerce.utils.ECommerceConstants
import android.content.Context
import com.google.gson.Gson
import kotlin.math.roundToInt

/**
 * Created by Moiz Hassan on 25, January,2022
 */

class BsLogCheckoutEvent {

    /**
     * BsLogCheckoutEvent is required to log checkout event for orders. You can trigger this event
     * when the order has been placed or unable to place. you need to provide orderId in both cases.
     * If you don't have the orderId in case of not being successful then you can pass cartId for
     * that order.
     */

    data class Builder(
        private var context: Context? = null,
        var order_id: String? = null,
        var cart_id: String? = null,
        var is_successful: Boolean = true,
        var price_value: Float? = null,
        var currency_value: String? = null,
        var item_list: ArrayList<ItemModel> = arrayListOf(),
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        private lateinit var checkoutObject : CheckoutObject

        /**
         * init is required to pass context to the SDK method and log the event.
         * if not provided, it will throw an exception on runtime.
         */
        fun init(context: Context) = apply { this.context = context }

        /**
         * setOrderId is required to log the orderId for the order being successful or failed
         * to place. In case of not having any orderId for failed cases you can pass the cartId
         * or a corresponding value that can be used to track in the catalog about the order
         * placement details.
         */
        fun setOrderId(order_id: String) = apply { this.order_id = order_id }

        /**
         * setCartId can be used to log the cartId the event is logged for. It is recommended to
         * include the unique cartId for the cart items so that they can be tracked.
         */
        fun setCartId(cart_id: String) = apply { this.cart_id = cart_id }

        /**
         * isSuccessful is required to log the successful placement of the order. False in case of
         * order placement is not successful.
         */
        fun isSuccessful(is_successful: Boolean) = apply { this.is_successful = is_successful }

        /**
         * setPrice is required to log the total price of the order being logged. Price format
         * should be in accordance to the currency selected.
         */
        fun setPrice(price: Float) = apply {
            this.price_value = ((price * 100.0).roundToInt() / 100.0).toFloat() }
        fun setPrice(price: Int?) = apply { this.price_value = price?.toFloat() }
        fun setPrice(price: Double) = apply {
            this.price_value = ((price * 100.0).roundToInt() / 100.0).toFloat() }

        /**
         * setCurrency is required to log the currency for for the order logged. Currency should
         * be in ISO 4217 format. For ease, SDK provides the enums to log the currency so that it
         * would be easy to log. You can also use the string function to provide the currency.
         * Below is the function for the logging currency using enum CurrencyCode.
         */
//        fun setCurrency(currency: CurrencyCode) = apply { this.currency_value = currency.name }

        /**
         * setCurrency is required to log the currency for for the order logged. Currency should
         * be in ISO 4217 format. For Ease, SDK provides the enums to log the currency so that it
         * would be easy to log. You can also use the string function to provide the currency.
         * Below is the function for the logging currency using String. Remember to use the same
         * strings as provided in the enums or else the event will be discarded.
         */
        fun setCurrency(currency: String) = apply {
            if (CoreConstants.enumContains<InternalCurrencyCode>(currency)) {
                this.currency_value = currency
            } else {
                ExceptionManager.throwEnumException("CurrencyCode")
            }
        }

        /**
         * addItem can be used to add the item being orders in to the checkout list. This log can
         * be used to add one item to the log at a time. Order item should be in a valid format.
         * With elements of the orderObject as:
         * ItemModel(itemID, type, price, currency, stock_status, quantity, promoId) Promo Id can be an
         * empty string or no value at all if the item does not have a promo offer that is obtained
         * by the user. You can add multiple addOrder functions to one checkout event to include
         * all the items in the order.
         */
        fun addItem(itemModel: ItemModel) = apply {
            ECommerceConstants.isItemValueObjectValid(itemModel)
            this.item_list.add(itemModel)
        }

        /**
         * setItem can be used to pass the whole item as a Json String object as well. You can use
         * the POJO ItemModel to parse the data int he required format and pass that to this
         * function as a string to log the event. You can use Gson to convert the object to string
         * but SDK will parse the Json string back to POJO so pass it in the log. This method
         * should be used with caution and is suitable for react native bridge.
         */
        fun addItem(itemJsonString: String) = apply {
            val item: ItemModel = Gson().fromJson(itemJsonString, ItemModel::class.java)
            ECommerceConstants.isItemValueObjectValid(item)
            this.item_list.add(item)
        }

        /**
         * addItemList can be used to add the whole list to the log at once, the format should be
         * ArrayList<ItemModel> to log the event successfully. Order item should be in a valid format.
         * With elements of the orderObject as:
         * ItemModel(itemID, type, price, currency, stock_status, quantity, promoId) Promo Id can be an
         * empty string or no value at all if the item does not have a promo offer that is obtained
         * by the user. You should use only one addItemList with checkout event or else the list
         * will only save the later one.
         */
        fun addItemList(itemList: ArrayList<ItemModel>) = apply {
            for(item in itemList){
                ECommerceConstants.isItemValueObjectValid(item)
            }
            this.item_list.addAll(itemList)
        }

        /**
         * addItemList can be used to add the whole list to the log at once, the format should be
         * ArrayList<ItemModel> to log the event successfully. But the input param is of type
         * string , this is special use case for react native logs where list can be passed as
         * Json string and can be used to log the event. Order item should be in a valid format.
         * With elements of the orderObject as:
         * ItemModel(itemID, type, price, currency, stock_status, quantity, promoId) Promo Id can be an
         * empty string or no value at all if the item does not have a promo offer that is obtained
         * by the user. You should use only one addItemList with checkout event or else the list
         * will only save the later one.
         */
        fun addItemList(itemListString: String) = apply {
            val itemModels: Array<ItemModel> = Gson().fromJson(
                itemListString,
                Array<ItemModel>::class.java
            )
            val itemList = java.util.ArrayList(itemModels.toMutableList())
            for(item in itemList){
                ECommerceConstants.isItemValueObjectValid(item)
            }
            this.item_list.addAll(itemList)
        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if context provided is null or not
                 * provided at all.
                 */
                context == null -> {
                    ExceptionManager.throwInitException()
                }

                /**
                 * Will throw and exception if the order_id provided is null or no value is
                 * provided at all.
                 */
                order_id.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("order_id")
                }

                /**
                 * Will throw and exception if the cart_id provided is null or no value is
                 * provided at all.
                 */
                cart_id.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("cart_id")
                }

                /**
                 * Will throw and exception if the price provided is null or no value is
                 * provided at all.
                 */
                price_value == null -> {
                    ExceptionManager.throwIsRequiredException("price_value")
                }

                /**
                 * Will throw and exception if the currency provided is null or no value is
                 * provided at all.
                 */
                currency_value.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(InternalCurrencyCode::class.java.simpleName)
                }

                /**
                 * Will throw and exception if the item_list provided is null or no value is
                 * provided at all for the order Items.
                 */
                item_list.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("item_list")
                }
                else -> {

                    for (item in item_list) {
                        if (currency_value != item.currency) {
                            ExceptionManager.throwCurrencyNotSameException("checkout")
                        }
                    }

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    checkoutObject = CheckoutObject(
                        order_id!!, cart_id!!, is_successful, price_value,
                        currency_value, item_list, null, meta
                    )
                    if (currency_value == InternalCurrencyCode.USD.name) {
                        checkoutObject.usd_rate = 1f
                        BenshiLog().track(
                            context!!, ECommerceConstants.contentBlockName, EComEventType.checkout.name,
                            checkoutObject, update_immediately
                        )
                    } else {
                        BenshiLog().getUSDRate(context!!, currency_value!!, ::getUSDRateAndLogEvent)
                    }
                }
            }
        }

        private fun getUSDRateAndLogEvent(usdRate : Float){
            checkoutObject.usd_rate = usdRate
            BenshiLog().track(
                context!!, ECommerceConstants.contentBlockName, EComEventType.checkout.name,
                checkoutObject, update_immediately
            )
        }

    }
}
