package ai.benshi.android.sdk.e_commerce.builders

import ai.benshi.android.sdk.core.BenshiLog
import ai.benshi.android.sdk.core.utils.CoreConstants
import ai.benshi.android.sdk.core.utils.ExceptionManager
import ai.benshi.android.sdk.e_commerce.event_models.DeliveryObject
import ai.benshi.android.sdk.e_commerce.event_types.DeliveryAction
import ai.benshi.android.sdk.e_commerce.event_types.EComEventType
import ai.benshi.android.sdk.e_commerce.utils.ECommerceConstants
import android.content.Context

/**
 * Created by Moiz Hassan on 01, March,2022
 */

class BsLogDeliveryEvent {

    /**
     * BsLogDeliveryEvent is used to log the status for the delivery. It can be used to log the
     * delivered status of the order or a partial order. Details about the items in the specific
     * delivery should be provided in the catalog.
     */
    data class Builder(
        private var context: Context? = null,
        var order_id: String? = null,
        var delivery_id: String? = null,
        var action: String? = null,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * init is required to pass context to the SDK method and log the event.
         * if not provided, it will throw an exception on runtime.
         */
        fun init(context: Context) = apply { this.context = context }

        /**
         * setOrderId is required to associate the rating obtained for the order. OrderId should
         * be a valid orderId and can be tracked from the catalog
         */
        fun setOrderId(order_id: String) = apply { this.order_id = order_id }

        /**
         * setDeliveryId is required to associate the rating obtained for the order. deliveryId should
         * be a valid deliveryId and can be tracked from the catalog for the items in that
         * specific delivery.
         */
        fun setDeliveryId(delivery_id: String) = apply { this.delivery_id = delivery_id }

        /**
         * setDeliveryAction is required to set the delivery action for the log. For the order
         * being prepared for delivery or left the shipment center or delivered to the customer.
         */
        fun setDeliveryAction(action: DeliveryAction) = apply { this.action = action.name }
        fun setDeliveryAction(action: String) = apply {
            if (CoreConstants.enumContains<DeliveryAction>(action)) {
                this.action = action
            } else {
                ExceptionManager.throwEnumException(DeliveryAction::class.java.simpleName)
            }
        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if context provided is null or not
                 * provided at all.
                 */
                context == null -> {
                    ExceptionManager.throwInitException()
                }

                /**
                 * Will throw and exception if the orderId provided is null or no value is
                 * provided at all.
                 */
                order_id.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("order_id")
                }

                /**
                 * Will throw and exception if the delivery_id provided is null or no value is
                 * provided at all.
                 */
                delivery_id.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("delivery_id")
                }

                /**
                 * Will throw and exception if the delivery_action provided is null or no value is
                 * provided at all.
                 */
                action.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(DeliveryAction::class.java.simpleName)
                }

                else -> {

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val deliveryObject = DeliveryObject(delivery_id!!, order_id!!, action!!, meta)
                    BenshiLog().track(
                        context!!, ECommerceConstants.contentBlockName, EComEventType.delivery.name,
                        deliveryObject, update_immediately
                    )
                }
            }
        }
    }
}
