package ai.benshi.android.sdk.e_commerce.builders

import ai.benshi.android.sdk.core.BenshiLog
import ai.benshi.android.sdk.core.utils.CoreConstants
import ai.benshi.android.sdk.core.utils.ExceptionManager
import ai.benshi.android.sdk.e_commerce.event_models.DeliveryObject
import ai.benshi.android.sdk.e_commerce.event_models.ScheduleDeliveryObject
import ai.benshi.android.sdk.e_commerce.event_types.DeliveryAction
import ai.benshi.android.sdk.e_commerce.event_types.EComEventType
import ai.benshi.android.sdk.e_commerce.event_types.ScheduleDeliveryAction
import ai.benshi.android.sdk.e_commerce.utils.ECommerceConstants
import android.content.Context
import java.text.SimpleDateFormat
import java.util.*

/**
 * Created by Moiz Hassan on 19, September, 2022
 */

class BsLogScheduleDeliveryEvent {

    /**
     * BsLogScheduleDeliveryEvent is used to log the status for the delivery. It can be used to log the
     * delivered status of the order or a partial order. Details about the items in the specific
     * delivery should be provided in the catalog.
     */
    data class Builder(
        private var context: Context? = null,
        internal var order_id: String? = null,
        internal var is_urgent: Boolean? = null,
        internal var action: String? = null,
        internal var delivery_ts: Long? = null,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * init is required to pass context to the SDK method and log the event.
         * if not provided, it will throw an exception on runtime.
         */
        fun init(context: Context) = apply { this.context = context }

        /**
         * setOrderId is required to associate the rating obtained for the order. OrderId should
         * be a valid orderId and can be tracked from the catalog
         */
        fun setOrderId(order_id: String) = apply { this.order_id = order_id }

        /**
         * setDeliveryId is required to associate the rating obtained for the order. deliveryId should
         * be a valid deliveryId and can be tracked from the catalog for the items in that
         * specific delivery.
         */
        fun isUrgent(is_urgent: Boolean) = apply { this.is_urgent = is_urgent }

        /**
         * setScheduleDeliveryAction is required to set the delivery action for the log. For the order
         * being prepared for delivery that if the item is being scheduled or updated in terms of
         * delivery elements
         */
        fun setScheduleDeliveryAction(action: ScheduleDeliveryAction) = apply { this.action = action.name }
        fun setScheduleDeliveryAction(action: String) = apply {
            if (CoreConstants.enumContains<ScheduleDeliveryAction>(action)) {
                this.action = action
            } else {
                ExceptionManager.throwEnumException(ScheduleDeliveryAction::class.java.simpleName)
            }
        }

        /**
         * The timestamp in RFC 3339 format for the date and time for which the delivery is set
         * to be scheduled
         */
        fun setDeliveryDateTime(delivery_ts: Long) = apply { this.delivery_ts = delivery_ts }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if context provided is null or not
                 * provided at all.
                 */
                context == null -> {
                    ExceptionManager.throwInitException()
                }

                /**
                 * Will throw and exception if the orderId provided is null or no value is
                 * provided at all.
                 */
                order_id.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("order_id")
                }

                /**
                 * Will throw and exception if the is_urgent provided is null or no value is
                 * provided at all.
                 */
                is_urgent == null -> {
                    ExceptionManager.throwIsRequiredException("is_urgent")
                }

                /**
                 * Will throw and exception if the delivery_action provided is null or no value is
                 * provided at all.
                 */
                action.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(ScheduleDeliveryAction::class.java.simpleName)
                }

                /**
                 * Will throw and exception if the delivery_ts provided is null or no value is
                 * provided at all.
                 */
                delivery_ts == null || delivery_ts == 0L -> {
                    ExceptionManager.throwIsRequiredException("delivery_ts")
                }

                else -> {

                    if(delivery_ts!! < System.currentTimeMillis()-10000L){
                        ExceptionManager.throwIllegalStateException("Scheduled time should be in future")
                    }

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */

                    val scheduleDeliveryObject = ScheduleDeliveryObject(
                        order_id!!, is_urgent!!, action!!,
                        ECommerceConstants.getDateTime(delivery_ts!!), meta)
                    BenshiLog().track(
                        context!!, ECommerceConstants.contentBlockName,
                        EComEventType.schedule_delivery.name,
                        scheduleDeliveryObject, update_immediately
                    )
                }
            }
        }
    }
}
