package ai.benshi.android.sdk.payments.builders

import ai.benshi.android.sdk.core.BenshiLog
import ai.benshi.android.sdk.core.event_types.EventType
import ai.benshi.android.sdk.core.utils.CoreConstants
import ai.benshi.android.sdk.payments.event_models.InvoiceListObject
import ai.benshi.android.sdk.payments.event_types.InvoiceListType
import ai.benshi.android.sdk.payments.utils.PaymentsConstants
import android.content.Context

class BsLogInvoiceListEvent {

    /**
     * BsLogInvoiceListEvent is to log list events for invoices in payments. There can be
     * multiple invoice types that depends on the status of invoice which can be either paid,
     * due or deposits.
     */
    data class Builder(
        private var context: Context? = null,
        private var type: String? = null,
        private var list_id: String? = null,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * init is required to pass context to the SDK method and log the event.
         * if not provided, it will throw an exception on runtime.
         */
        fun init(context: Context) = apply { this.context = context }

        /**
         * setInvoiceListType is used to specify the type of Invoice List, there can be 3 types of
         * invoice list types, due, paid and deposits. By default SDK provides enum class to use as
         * InvoiceListType. Below function can be used to set list type using enum.
         */
        fun setInvoiceListType(type: InvoiceListType) = apply { this.type = type.name }

        /**
         * setInvoiceListType is used to specify the type of Invoice List, there can be 3 types of
         * invoice list types, due, paid and deposits. By default SDK provides enum class to use as
         * InvoiceListType. Below function can be used to set list type using string but remember
         * to note that string provided must be in accordance to the enums.
         */
        fun setInvoiceListType(type: String) = apply {
            if (CoreConstants.enumContains<InvoiceListType>(type)) {
                this.type = type
            } else {
                throw IllegalArgumentException("Invalid Invoice_list_type provided")
            }
        }

        /**
         * setListId is required to set the id for the invoice list in log, Id should be in string
         * and must be in accordance to the catalog provided. if you don't have an id, you can
         * either skip this or pass null.
         */
        fun setListId(list_id: String?) = apply { this.list_id = list_id }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if context provided is null or not
                 * provided at all.
                 */
                context == null -> {
                    throw NullPointerException("init is required to provide context.")
                }

                /**
                 * Will throw and exception if the invoiceType provided is null or no value is
                 * provided at all.
                 */
                type == null -> {
                    throw NullPointerException("invoiceType is required.")
                }

                else -> {

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val invoiceListObject = InvoiceListObject(type!!, list_id, meta)
                    BenshiLog.track(
                        context!!, PaymentsConstants.moduleName, EventType.invoice_list,
                        invoiceListObject, update_immediately
                    )
                }
            }
        }
    }
}