package ai.benshi.android.sdk.payments.builders

import ai.benshi.android.sdk.core.BenshiLog
import ai.benshi.android.sdk.core.event_types.CurrencyCode
import ai.benshi.android.sdk.core.event_types.EventType
import ai.benshi.android.sdk.core.utils.CoreConstants
import ai.benshi.android.sdk.payments.event_models.InvoiceObject
import ai.benshi.android.sdk.payments.event_models.PaymentObject
import ai.benshi.android.sdk.payments.event_types.PaymentAction
import ai.benshi.android.sdk.payments.event_types.PaymentType
import ai.benshi.android.sdk.payments.utils.PaymentsConstants
import android.content.Context

class BsLogPaymentEvent {

    /**
     * BsLogPaymentEvent is to log events for payments, when they undergo processing and
     * processed completed or canceled.
     */
    data class Builder(
        private var context: Context? = null,
        private var payment_id: String? = null,
        private var action: String? = null,
        private var type: String? = null,
        private var currency: String? = null,
        private var amount: Float? = null,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * init is required to pass context to the SDK method and log the event.
         * if not provided, it will throw an exception on runtime.
         */
        fun init(context: Context) = apply { this.context = context }

        /**
         * setPaymentId is required to set the id for the payment in log, Id should be in string
         * and must be in accordance to the catalog provided.
         */
        fun setPaymentId(payment_id: String) = apply { this.payment_id = payment_id }

        /**
         * setPaymentAction is used to specify the action performed on the payments screen,
         * there can be multiple types of actions that can include init payment process, cancel
         * payment processing or upload receipt to bank deposit. By default SDK provides enum
         * class to use as PaymentAction. Below function can be used to set action using enum.
         */
        fun setPaymentAction(action: PaymentAction) = apply { this.action = action.name }

        /**
         * setPaymentAction is used to specify the action performed on the payments screen,
         * there can be multiple types of actions that can include init payment process, cancel
         * payment processing or upload receipt to bank deposit. By default SDK provides enum
         * class to use as PaymentAction. Below function can be used to set action using string
         * but remember to note that string provided must be in accordance to the enums.
         */
        fun setPaymentAction(action: String) = apply {
            if (CoreConstants.enumContains<PaymentAction>(action)) {
                this.action = action
            } else {
                throw IllegalArgumentException("Invalid payment_action provided")
            }
        }

        /**
         * setPaymentType is used to specify the type of payment, does it involve actual money
         * which is referred as physical money and the other is in-app currency which is referred
         * as virtual. Below function can be used to set type using enum.
         */
        fun setPaymentType(type: PaymentType) = apply { this.type = type.name }

        /**
         * setPaymentType is used to specify the type of payment, does it involve actual money
         * which is referred as physical money and the other is in-app currency which is referred
         * as virtual. Below function can be used to set type using string but remember to note
         * that string provided must be in accordance to the enums.
         */
        fun setPaymentType(type: String) = apply {
            if (CoreConstants.enumContains<PaymentType>(type)) {
                this.type = type
            } else {
                throw IllegalArgumentException("Invalid payment_type provided")
            }
        }

        /**
         * setCurrency is required to log the currency for for the payment logged. Currency should
         * be in ISO 4217 format. For ease, SDK provides the enums to log the currency so that it
         * would be easy to log. You can also use the string function to provide the currency.
         * Below is the function for the logging currency using enum CurrencyCode.
         */
        fun setCurrency(currency: CurrencyCode?) = apply { this.currency = currency?.name }

        /**
         * setCurrency is required to log the currency for for the payment logged. Currency should
         * be in ISO 4217 format. For Ease, SDK provides the enums to log the currency so that it
         * would be easy to log. You can also use the string function to provide the currency.
         * Below is the function for the logging currency using String. Remember to use the same
         * strings as provided in the enums or else the event will be discarded.
         */
        fun setCurrency(currency: String?) = apply {
            if(!currency.isNullOrEmpty()){
                if (CoreConstants.enumContains<CurrencyCode>(currency)) {
                    this.currency = currency
                } else {
                    throw IllegalArgumentException("Invalid currency_code provided")
                }
            }
        }

        /**
         * setAmount is required to log the total price of the payment being logged. Amount format
         * should be in accordance to the currency selected.
         */
        fun setAmount(amount: Float?) = apply { this.amount = amount }
        fun setAmount(amount: Int?) = apply { this.amount = amount?.toFloat() }
        fun setAmount(amount: Double?) = apply { this.amount = amount?.toFloat() }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if context provided is null or not
                 * provided at all.
                 */
                context == null -> {
                    throw NullPointerException("init is required to provide context.")
                }

                /**
                 * Will throw and exception if the paymentId provided is null or no value is
                 * provided at all.
                 */
                payment_id == null -> {
                    throw NullPointerException("paymentId is required.")
                }

                /**
                 * Will throw and exception if the paymentAction provided is null or no value is
                 * provided at all.
                 */
                action == null -> {
                    throw NullPointerException("paymentAction is required.")
                }

                /**
                 * Will throw and exception if the paymentType provided is null or no value is
                 * provided at all.
                 */
                type == null -> {
                    throw NullPointerException("paymentType is required.")
                }

                else -> {

                    val isValidType = action != PaymentAction.send_invoice.name &&
                            action != PaymentAction.join_chat.name &&
                            action != PaymentAction.copy_payment_detail.name &&
                            action != PaymentAction.set_default_payment.name
                    if(isValidType && currency == null) {
                        throw NullPointerException("currency is required.")
                    }else if(isValidType && amount == null){
                        throw NullPointerException("amount is required.")
                    }else{
                        /**
                         * Parsing the values into an object and passing to the setup block to queue
                         * the event based on its priority.
                         */
                        val paymentObject = PaymentObject(payment_id!!, action!!, type!!, currency,
                            amount, meta)
                        BenshiLog.track(
                            context!!, PaymentsConstants.moduleName, EventType.payment,
                            paymentObject, update_immediately
                        )
                    }

                }
            }
        }
    }
}