package ai.benshi.android.sdk.payments.builders

import ai.benshi.android.sdk.core.BenshiLog
import ai.benshi.android.sdk.core.utils.CoreConstants
import ai.benshi.android.sdk.payments.event_models.InvoiceObject
import ai.benshi.android.sdk.payments.event_types.InvoiceAction
import ai.benshi.android.sdk.payments.event_types.PaymentsEventType
import ai.benshi.android.sdk.payments.utils.PaymentsConstants
import android.content.Context

class BsLogInvoiceEvent {

    /**
     * BsLogInvoiceEvent is to log events for invoices These events can be viewing an invoice or
     * viewing the detail of the invoice.
     */
    data class Builder(
        private var context: Context? = null,
        private var action: String? = null,
        private var invoice_id: String? = null,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * init is required to pass context to the SDK method and log the event.
         * if not provided, it will throw an exception on runtime.
         */
        fun init(context: Context) = apply { this.context = context }

        /**
         * setInvoiceAction is used to specify the action performed on the invoice, there can be
         * 2 types of actions, view and detail. By default SDK provides enum class to use as
         * InvoiceAction. Below function can be used to set action using enum.
         */
        fun setInvoiceAction(action: InvoiceAction) = apply { this.action = action.name }

        /**
         * setInvoiceAction is used to specify the action performed on the invoice, there can be
         * 2 types of actions, view and detail. By default SDK provides enum class to use as
         * InvoiceAction. Below function can be used to set action using string but remember
         * to note that string provided must be in accordance to the enums.
         */
        fun setInvoiceAction(action: String) = apply {
            if (CoreConstants.enumContains<InvoiceAction>(action)) {
                this.action = action
            } else {
                throw IllegalArgumentException("Invalid Invoice_action provided")
            }
        }

        /**
         * setInvoiceId is required to set the id for the invoice in log, Id should be in string
         * and must be in accordance to the catalog provided.
         */
        fun setInvoiceId(invoice_id: String) = apply { this.invoice_id = invoice_id }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if context provided is null or not
                 * provided at all.
                 */
                context == null -> {
                    throw NullPointerException("init is required to provide context.")
                }

                /**
                 * Will throw and exception if the invoiceAction provided is null or no value is
                 * provided at all.
                 */
                action == null -> {
                    throw NullPointerException("invoiceAction is required.")
                }

                /**
                 * Will throw and exception if the invoiceId provided is null or no value is
                 * provided at all.
                 */
                invoice_id == null -> {
                    throw NullPointerException("invoiceId is required.")
                }

                else -> {

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val invoiceObject = InvoiceObject(invoice_id!!, action!!, meta)
                    BenshiLog.track(
                        context!!, PaymentsConstants.moduleName, PaymentsEventType.invoice.name,
                        invoiceObject, update_immediately
                    )
                }
            }
        }
    }
}