package ai.benshi.android.sdk.payments.builders

import ai.benshi.android.sdk.core.BenshiLog
import ai.benshi.android.sdk.core.utils.CoreConstants
import ai.benshi.android.sdk.payments.event_models.UpdateBalanceObject
import ai.benshi.android.sdk.payments.event_types.BalanceSource
import ai.benshi.android.sdk.payments.event_types.CurrencyCode
import ai.benshi.android.sdk.payments.event_types.PaymentsEventType
import ai.benshi.android.sdk.payments.event_types.UpdateBalanceType
import ai.benshi.android.sdk.payments.utils.PaymentsConstants
import android.content.Context

class BsLogUpdateBalanceEvent {

    /**
     * BsLogUpdateBalanceEvent is to log events for updating user in app balance, can be credit or
     * debit which may or may not depict the usage or e-wallet.
     */
    data class Builder(
        private var context: Context? = null,
        private var type: String? = null,
        private var source: String? = null,
        private var currency: String? = null,
        private var amount: Float? = null,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * init is required to pass context to the SDK method and log the event.
         * if not provided, it will throw an exception on runtime.
         */
        fun init(context: Context) = apply { this.context = context }

        /**
         * setUpdateType is used to specify the type of the update, there can be
         * 2 types of updates, credit and debit. By default SDK provides enum class to use as
         * UpdateBalanceType. Below function can be used to set action using enum.
         */
        fun setUpdateType(type: UpdateBalanceType) = apply { this.type = type.name }

        /**
         * setUpdateType is used to specify the type of the update, there can be
         * 2 types of updates, credit and debit. By default SDK provides enum class to use as
         * UpdateBalanceType. Below function can be used to set action using string but remember
         * to note that string provided must be in accordance to the enums.
         */
        fun setUpdateType(type: String) = apply {
            if (CoreConstants.enumContains<UpdateBalanceType>(type)) {
                this.type = type
            } else {
                throw IllegalArgumentException("Invalid Update_Balance_Type provided")
            }
        }

        /**
         * setSourceType is used to specify the source of the balance, there can be
         * 3 types: deposit, voucher and points. By default SDK provides enum class to use as
         * BalanceSource. Source is only required when you credit your balance. Below function
         * can be used to set action using enum.
         */
        fun setSourceType(source: BalanceSource) = apply { this.source = source.name }

        /**
         * setSourceType is used to specify the source of the balance, there can be
         * 3 types: deposit, voucher and points. By default SDK provides enum class to use as
         * BalanceSource. source is only required when you credit your balance. Below function
         * can be used to set action using string but remember to note that string provided must
         * be in accordance to the enums.
         */
        fun setSourceType(source: String) = apply {
            if (CoreConstants.enumContains<BalanceSource>(source)) {
                this.source = source
            } else {
                throw IllegalArgumentException("Invalid Balance_Source provided")
            }
        }

        /**
         * setCurrency is required to log the currency for for the balance logged. Currency should
         * be in ISO 4217 format. For ease, SDK provides the enums to log the currency so that it
         * would be easy to log. You can also use the string function to provide the currency.
         * Below is the function for the logging currency using enum CurrencyCode.
         */
        fun setCurrency(currency: CurrencyCode?) = apply { this.currency = currency?.name }

        /**
         * setCurrency is required to log the currency for for the balance logged. Currency should
         * be in ISO 4217 format. For Ease, SDK provides the enums to log the currency so that it
         * would be easy to log. You can also use the string function to provide the currency.
         * Below is the function for the logging currency using String. Remember to use the same
         * strings as provided in the enums or else the event will be discarded.
         */
        fun setCurrency(currency: String?) = apply {
            if(!currency.isNullOrEmpty()){
                if (CoreConstants.enumContains<CurrencyCode>(currency)) {
                    this.currency = currency
                } else {
                    throw IllegalArgumentException("Invalid currency_code provided")
                }
            }
        }

        /**
         * setAmount is required to log the total price of the balance being logged. Amount format
         * should be in accordance to the currency selected.
         */
        fun setAmount(amount: Float?) = apply { this.amount = amount }
        fun setAmount(amount: Int?) = apply { this.amount = amount?.toFloat() }
        fun setAmount(amount: Double?) = apply { this.amount = amount?.toFloat() }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if context provided is null or not
                 * provided at all.
                 */
                context == null -> {
                    throw NullPointerException("init is required to provide context.")
                }

                /**
                 * Will throw and exception if the UpdateBalanceType provided is null or no value is
                 * provided at all.
                 */
                type == null -> {
                    throw NullPointerException("UpdateBalanceType is required.")
                }

                /**
                 * Will throw and exception if the currency provided is null or no value is
                 * provided at all.
                 */
                currency == null -> {
                    throw NullPointerException("currency is required.")
                }

                /**
                 * Will throw and exception if the amount provided is null or no value is
                 * provided at all.
                 */
                amount == null -> {
                    throw NullPointerException("amount is required.")
                }

                else -> {

                    if(type == UpdateBalanceType.credit.name && source == null){
                        throw NullPointerException("Balance source is required.")
                    }
                    else{
                        /**
                         * Parsing the values into an object and passing to the setup block to queue
                         * the event based on its priority.
                         */
                        val virtualCreditObject = UpdateBalanceObject(type!!, source, currency!!, amount!!, meta)
                        BenshiLog.track(
                            context!!, PaymentsConstants.moduleName,
                            PaymentsEventType.update_balance.name,
                            virtualCreditObject, update_immediately
                        )
                    }
                }
            }
        }
    }
}