package ai.benshi.android.sdk.payments.builders

import ai.benshi.android.sdk.core.BenshiLog
import ai.benshi.android.sdk.core.utils.CoreConstants
import ai.benshi.android.sdk.core.utils.ExceptionManager
import ai.benshi.android.sdk.payments.event_models.DeferredPaymentObject
import ai.benshi.android.sdk.payments.event_models.InternalCurrencyCode
import ai.benshi.android.sdk.payments.event_types.PaymentAction
import ai.benshi.android.sdk.payments.event_types.PaymentsEventType
import ai.benshi.android.sdk.payments.utils.PaymentsConstants
import android.content.Context
import kotlin.math.roundToInt

class BsLogDeferredPaymentEvent {

    /**
     * BsLogDeferredPaymentEvent is to log events for payments, when they undergo processing and
     * processed completed or canceled. This log is for use cases whe we have Buy Now and Pay Later
     * which means that customers can buy the items now and pay the amount on a later date.
     */
    data class Builder(
        private var context: Context? = null,
        var order_id: String? = null,
        var payment_id: String? = null,
        var action: String? = null,
        private var account_balance: Float? = null,
        var payment_amount: Float? = null,
        var currency_value: String? = null,
        private var is_successful: Boolean = true,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        private lateinit var paymentObject : DeferredPaymentObject

        /**
         * init is required to pass context to the SDK method and log the event.
         * if not provided, it will throw an exception on runtime.
         */
        fun init(context: Context) = apply { this.context = context }

        /**
         * setOrderId is required to set the id for the order in log, Id should be in string
         * and must be in accordance to the catalog provided.
         */
        fun setOrderId(order_id: String) = apply { this.order_id = order_id }

        /**
         * setPaymentId is required to set the id for the payment in log, Id should be in string
         * and must be in accordance to the catalog provided.
         */
        fun setPaymentId(payment_id: String) = apply { this.payment_id = payment_id }

        /**
         * setPaymentAction is used to specify the action performed on the payments screen,
         * there can be multiple types of actions that can include init payment process, cancel
         * payment processing or upload receipt to bank deposit. By default SDK provides enum
         * class to use as PaymentAction. Below function can be used to set action using enum.
         */
        fun setPaymentAction(action: PaymentAction) = apply { this.action = action.name }

        /**
         * setPaymentAction is used to specify the action performed on the payments screen,
         * there can be multiple types of actions that can include init payment process, cancel
         * payment processing or upload receipt to bank deposit. By default SDK provides enum
         * class to use as PaymentAction. Below function can be used to set action using string
         * but remember to note that string provided must be in accordance to the enums.
         */
        fun setPaymentAction(action: String) = apply {
            if (CoreConstants.enumContains<PaymentAction>(action)) {
                this.action = action
            } else {
                ExceptionManager.throwEnumException(PaymentAction::class.java.simpleName)
            }
        }

        /**
         * setCurrency is required to log the currency for for the payment logged. Currency should
         * be in ISO 4217 format. For ease, SDK provides the enums to log the currency so that it
         * would be easy to log. You can also use the string function to provide the currency.
         * Below is the function for the logging currency using enum CurrencyCode.
         */
//        fun setCurrency(currency: CurrencyCode?) = apply { this.currency_value = currency?.name }

        /**
         * setCurrency is required to log the currency for for the payment logged. Currency should
         * be in ISO 4217 format. For Ease, SDK provides the enums to log the currency so that it
         * would be easy to log. You can also use the string function to provide the currency.
         * Below is the function for the logging currency using String. Remember to use the same
         * strings as provided in the enums or else the event will be discarded.
         */
        fun setCurrency(currency: String?) = apply {
            if (!currency.isNullOrEmpty()) {
                if (CoreConstants.enumContains<InternalCurrencyCode>(currency)) {
                    this.currency_value = currency
                } else {
                    ExceptionManager.throwEnumException("CurrencyCode")
                }
            }
        }

        /**
         * setAccountBalance is required to log the current balance of the user. Amount format
         * should be in accordance to the currency selected.
         */
        fun setAccountBalance(account_balance: Float) = apply {
            this.account_balance = ((account_balance * 100.0).roundToInt() / 100.0).toFloat()
        }

        fun setAccountBalance(account_balance: Int?) =
            apply { this.account_balance = account_balance?.toFloat() }

        fun setAccountBalance(account_balance: Double) = apply {
            this.account_balance = ((account_balance * 100.0).roundToInt() / 100.0).toFloat()
        }

        /**
         * setPaymentAmount is required to log the total price of the payment being logged. Amount
         * format should be in accordance to the currency selected.
         */
        fun setPaymentAmount(payment_amount: Float) = apply {
            this.payment_amount = ((payment_amount * 100.0).roundToInt() / 100.0).toFloat()
        }

        fun setPaymentAmount(payment_amount: Int?) =
            apply { this.payment_amount = payment_amount?.toFloat() }

        fun setPaymentAmount(payment_amount: Double) = apply {
            this.payment_amount = ((payment_amount * 100.0).roundToInt() / 100.0).toFloat()
        }

        /**
         * isSuccessful is required to log if the payment is processed successfully or not.
         */
        fun isSuccessful(is_successful: Boolean) = apply { this.is_successful = is_successful }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if context provided is null or not
                 * provided at all.
                 */
                context == null -> {
                    ExceptionManager.throwInitException()
                }

                /**
                 * Will throw and exception if the orderId provided is null or no value is
                 * provided at all.
                 */
                order_id == null -> {
                    ExceptionManager.throwIsRequiredException("order_id")
                }

                /**
                 * Will throw and exception if the paymentId provided is null or no value is
                 * provided at all.
                 */
                payment_id == null -> {
                    ExceptionManager.throwIsRequiredException("payment_id")
                }

                /**
                 * Will throw and exception if the paymentAction provided is null or no value is
                 * provided at all.
                 */
                action == null -> {
                    ExceptionManager.throwIsRequiredException(PaymentAction::class.java.simpleName)
                }

                /**
                 * Will throw and exception if the account_balance provided is null or no value is
                 * provided at all.
                 */
                account_balance == null -> {
                    ExceptionManager.throwIsRequiredException("account_balance")
                }

                /**
                 * Will throw and exception if the payment_amount provided is null or no value is
                 * provided at all.
                 */
                payment_amount == null -> {
                    ExceptionManager.throwIsRequiredException("payment_amount")
                }

                /**
                 * Will throw and exception if the currency provided is null or no value is
                 * provided at all.
                 */
                currency_value == null -> {
                    ExceptionManager.throwIsRequiredException(InternalCurrencyCode::class.java.simpleName)
                }

                else -> {
                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    paymentObject = DeferredPaymentObject(
                        payment_id!!, order_id!!, action!!, account_balance!!,
                        payment_amount!!, currency_value, is_successful, null, meta
                    )
                    if (currency_value == InternalCurrencyCode.USD.name) {
                        paymentObject.usd_rate = 1f
                        BenshiLog().track(
                            context!!, PaymentsConstants.contentBlockName, PaymentsEventType.deferred_payments.name,
                            paymentObject, update_immediately
                        )
                    } else {
                        BenshiLog().getUSDRate(context!!, currency_value!!, ::getUSDRateAndLogEvent)
                    }
                }
            }
        }

        private fun getUSDRateAndLogEvent(usdRate : Float){
            paymentObject.usd_rate = usdRate
            BenshiLog().track(
                context!!, PaymentsConstants.contentBlockName, PaymentsEventType.deferred_payments.name,
                paymentObject, update_immediately
            )
        }

    }
}
