package ai.causalfoundry.android.sdk.e_commerce.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_commerce.catalog.CfEComCatalog
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.BloodCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.DrugCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.MedicalEquipmentCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.OxygenCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.InternalCurrencyCode
import ai.causalfoundry.android.sdk.e_commerce.event_models.ItemModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.ViewItemObject
import ai.causalfoundry.android.sdk.e_commerce.event_types.EComEventType
import ai.causalfoundry.android.sdk.e_commerce.event_types.ItemAction
import ai.causalfoundry.android.sdk.e_commerce.event_types.ItemStockStatus
import ai.causalfoundry.android.sdk.e_commerce.event_types.ItemType
import ai.causalfoundry.android.sdk.e_commerce.utils.ECommerceConstants
import com.google.gson.Gson
import kotlin.math.roundToInt

/**
 * Created by Moiz Hassan on 22 March, 2023
 */

class CfLogItemEvent {

    /**
     * CfLogItemEvent is required to log item related events which included when an item is viewed
     * and when an item's detail is viewed.
     */

    data class Builder(
        var item_action_value: String? = null,
        var item_value: ItemModel = ItemModel("", 1, -1f, "", "", "", "", null),
        var search_id: String? = null,
        var catalog_model: Any? = null,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {
        private lateinit var itemObject: ViewItemObject

        /**
         * setItemViewAction is required to set Type for the type of log in this case, if a user is viewing
         * an item or it's details. setType log is used to define if the log is about the item
         * itself or if the user id going into the details. you can provide the values using 2
         * different methods, one is for enum based and the other is for string based.
         * Below is the method for the enum based approach.
         */
        fun setItemAction(item_actionValue: ItemAction) =
            apply { this.item_action_value = item_actionValue.name }

        /**
         * setItemViewAction is required to set Type for the type of log in this case, if a user is viewing
         * an item or it's details. setType log is used to define if the log is about the item
         * itself or if the user id going into the details. you can provide the values using 2
         * different methods, one is for enum based and the other is for string based.
         * Below is the method for the string based approach. Remember to note that for string
         * input you need to use the same names as defined in the enum or else the events will
         * be discarded.
         */
        fun setItemAction(item_actionValue: String) = apply {
            if (CoreConstants.enumContains<ItemAction>(item_actionValue)) {
                this.item_action_value = item_actionValue
            } else {
                ExceptionManager.throwEnumException(ItemAction::class.java.simpleName)
            }
        }

        /**
         * setItemId is required to log the Id for item the events are being logged. Details
         * about the item are to be provided in the catalog for more details about the item.
         */
        fun setItemId(item_id: String) = apply {
            this.item_value.id = item_id
        }

        /**
         * setItemQuantity is required to log the quantity for item the events are being logged. Details
         * about the item are to be provided in the catalog for more details about the item.
         */
        fun setItemQuantity(item_quantity: Int) = apply {
            this.item_value.quantity = item_quantity
        }

        /**
         * setItemPrice is required to log the price for item the events are being logged. Details
         * about the item are to be provided in the catalog for more details about the item.
         */
        fun setItemPrice(item_price: Float) = apply {
            this.item_value.price = ((item_price * 100.0).roundToInt() / 100.0).toFloat()
        }

        fun setItemPrice(item_price: Int) = apply { this.item_value.price = item_price.toFloat() }
        fun setItemPrice(item_price: Double) = apply {
            this.item_value.price = ((item_price * 100.0).roundToInt() / 100.0).toFloat()
        }

        /**
         * setItemCurrency is required to log the currency for item the events are being logged. Details
         * about the item are to be provided in the catalog for more details about the item.
         */
//        fun setItemCurrency(currency_code: CurrencyCode) = apply { this.item_currency = currency_code.name }
        fun setItemCurrency(currency_code: String) = apply {
            if (CoreConstants.enumContains<InternalCurrencyCode>(currency_code)) {
                this.item_value.currency = currency_code
            } else {
                ExceptionManager.throwEnumException("CurrencyCode")
            }
        }

        /**
         * setItemType is required to log the type for item the events are being
         * logged. Details about the item are to be provided in the catalog for more details about
         * the item.
         */
        fun setItemType(item_type: ItemType) = apply { this.item_value.type = item_type.name }
        fun setItemType(item_type: String) = apply {
            if (CoreConstants.enumContains<ItemType>(item_type)) {
                this.item_value.type = item_type
            } else {
                ExceptionManager.throwEnumException(ItemType::class.java.simpleName)
            }
        }

        /**
         * setItemStockStatus is required to log the stock status for item the events are being
         * logged. Details about the item are to be provided in the catalog for more details about
         * the item.
         */
        fun setItemStockStatus(stock_status: ItemStockStatus) = apply {
            this.item_value.stockStatus = stock_status.name
        }

        fun setItemStockStatus(stock_status: String) = apply {
            if (CoreConstants.enumContains<ItemStockStatus>(stock_status)) {
                this.item_value.stockStatus = stock_status
            } else {
                ExceptionManager.throwEnumException(ItemStockStatus::class.java.simpleName)
            }
        }

        /**
         * setItemPromoId is required to log the promo Id for item the events are being
         * logged. Details about the item are to be provided in the catalog for more details about
         * the item. If there is not promo associated with the product, you do not need to pass it.
         */
        fun setItemPromoId(promo_id: String) = apply { this.item_value.promoId = promo_id }

        /**
         * setItem can be used to pass the whole item as an object as well. You can use the POJO
         * ItemModel to parse the data int he required format and pass that to this function to
         * log the event.
         */
        fun setItem(item: ItemModel) = apply {
            this.item_value = item
        }

        /**
         * setItem can be used to pass the whole item as a Json String object as well. You can use
         * the POJO ItemModel to parse the data int he required format and pass that to this
         * function as a string to log the event. You can use Gson to convert the object to string
         * but SDK will parse the Json string back to POJO so pass it in the log. This method
         * should be used with caution and is suitable for react native bridge.
         */
        fun setItem(itemJsonString: String) = apply {
            val item: ItemModel = Gson().fromJson(itemJsonString, ItemModel::class.java)
            this.item_value = item
        }

        /**
         * setSearchId is used to associate the search id with the item being viewed by the user.
         * It is required to track if the item is a result of some search performed by the user in
         * the app.
         */
        fun setSearchId(search_id: String?) = apply { this.search_id = search_id }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }


        fun setCatalogProperties(catalogProperties: Any?) =
            apply {
                if (catalogProperties != null && catalogProperties != "") {

                    try {
                        if (catalogProperties is String) {
                            when (item_value.type) {
                                ItemType.drug.name -> {
                                    this.catalog_model = Gson().fromJson(
                                        catalogProperties,
                                        DrugCatalogModel::class.java
                                    )
                                }
                                ItemType.blood.name -> {
                                    this.catalog_model = Gson().fromJson(
                                        catalogProperties,
                                        BloodCatalogModel::class.java
                                    )
                                }
                                ItemType.oxygen.name -> {
                                    this.catalog_model = Gson().fromJson(
                                        catalogProperties,
                                        OxygenCatalogModel::class.java
                                    )
                                }
                                ItemType.medical_equipment.name -> {
                                    this.catalog_model = Gson().fromJson(
                                        catalogProperties,
                                        MedicalEquipmentCatalogModel::class.java
                                    )
                                }
                                else -> {
                                    ExceptionManager.throwIllegalStateException(
                                        "Please use correct catalog properties with provided item type"
                                    )
                                }
                            }
                        } else {
                            this.catalog_model = catalogProperties
                        }
                    } catch (ex: Exception) {
                        ExceptionManager.throwIllegalStateException(
                            "Please use correct catalog properties with provided item type"
                        )
                    }
                }
            }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw and exception if the type provided is null or no value is
                 * provided at all.
                 */
                item_action_value.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(ItemAction::class.java.simpleName)
                }

                else -> {
                    ECommerceConstants.isItemValueObjectValid(item_value, EComEventType.item)

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */

                    itemObject =
                        ViewItemObject(item_action_value!!, item_value, search_id, null, meta)

                    if (item_value.currency == InternalCurrencyCode.USD.name) {
                        itemObject.usd_rate = 1f
                        CFSetup().track(
                            ECommerceConstants.contentBlockName, EComEventType.item.name,
                            itemObject, update_immediately
                        )
                    } else {
                        CFSetup().getUSDRate(
                            item_value.currency,
                            ::getUSDRateAndLogEvent
                        )
                    }

                    if (item_action_value == ItemAction.view.name && this.catalog_model != null) {
                        CfEComCatalog.callCatalogAPI(
                            itemId = item_value.id,
                            itemtype = item_value.type,
                            catalogModel = this.catalog_model!!
                        )
                    }

                }
            }
        }

        private fun getUSDRateAndLogEvent(usdRate: Float) {
            itemObject.usd_rate = usdRate
            CFSetup().track(
                ECommerceConstants.contentBlockName, EComEventType.item.name,
                itemObject, update_immediately
            )
        }

    }
}
