package ai.causalfoundry.android.sdk.e_commerce.impression_listerner

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.impression_listener.BshCoreImpressionListener
import ai.causalfoundry.android.sdk.core.impression_listener.impression_model.RecyclerImpressionModel
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.core.utils.serializeToMap
import ai.causalfoundry.android.sdk.core.utils.toDataClass
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.BloodCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.DrugCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.MedicalEquipmentCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.OxygenCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.InternalCurrencyCode
import ai.causalfoundry.android.sdk.e_commerce.event_models.ViewItemObject
import ai.causalfoundry.android.sdk.e_commerce.event_types.EComEventType
import ai.causalfoundry.android.sdk.e_commerce.event_types.ItemAction
import ai.causalfoundry.android.sdk.e_commerce.event_types.ItemType
import ai.causalfoundry.android.sdk.e_commerce.impression_listerner.event_model.ItemImpressionModel
import ai.causalfoundry.android.sdk.e_commerce.utils.ECommerceConstants
import androidx.recyclerview.widget.RecyclerView
import com.google.gson.Gson

/**
 * Created by Moiz Hassan on 22 March, 2023
 */


object CfItemImpressionListener {

    private var usdRateValue: Float = 1f
    private lateinit var recyclerViewValue: RecyclerView
    private lateinit var currentDataProviderValue: List<ItemImpressionModel>
    private var searchIdValue: String = ""
    private var collectionViewId: String = ""

    @JvmStatic
    fun trackRecyclerView(
        recyclerView: RecyclerView,
        currentDataProvider: () -> List<ItemImpressionModel>,
        searchId: String
    ) {

        recyclerViewValue = recyclerView
        currentDataProviderValue = currentDataProvider()
        searchIdValue = searchId

        if (currentDataProvider()[0].itemProperties.currency != InternalCurrencyCode.USD.name) {
            CFSetup().getUSDRate(
                currentDataProvider()[0].itemProperties.currency,
                CfItemImpressionListener::getUSDRateAndLogEvent
            )
        } else {
            callCoreImpressionListenerTrackRecyclerView()
        }

    }

    // for native SDK  to track elements in a recyclerview
    private fun callCoreImpressionListenerTrackRecyclerView() {
        BshCoreImpressionListener.trackRecyclerView(recyclerViewValue, currentDataProvider = {
            currentDataProviderValue.map {
                RecyclerImpressionModel(
                    elementId = it.itemProperties.id,
                    contentBlock = ECommerceConstants.contentBlockName,
                    eventName = EComEventType.item.name,
                    catalogSubject = it.itemProperties.type,
                    itemProperties = ViewItemObject(
                        ItemAction.impression.name,
                        it.itemProperties, searchIdValue, usdRateValue, null
                    ),
                    catalogProperties = prepareCatalogObject(
                        it.itemProperties.id,
                        it.itemProperties.type,
                        it.catalogProperties
                    )
                )
            }
        }, searchIdValue, null)
    }

    private fun getUSDRateAndLogEvent(usdRate: Float) {
        usdRateValue = usdRate
        callCoreImpressionListenerTrackRecyclerView()
    }

    private fun prepareCatalogObject(
        itemId: String,
        itemType: String,
        catalogModel: Any?
    ): Any? {
        return when (itemType) {
            ItemType.drug.name -> {
                ECommerceConstants.verifyCatalogForDrug(itemId, catalogModel as DrugCatalogModel)
            }
            ItemType.blood.name -> {
                ECommerceConstants.verifyCatalogForBlood(
                    itemId,
                    catalogModel as BloodCatalogModel
                )
            }
            ItemType.oxygen.name -> {
                ECommerceConstants.verifyCatalogForOxygen(
                    itemId,
                    catalogModel as OxygenCatalogModel
                )
            }
            ItemType.medical_equipment.name -> {
                ECommerceConstants.verifyCatalogForMedicalEquipment(
                    itemId,
                    catalogModel as MedicalEquipmentCatalogModel
                )
            }
            else -> {
                null
            }
        }

    }

    interface CurrentDataProvider {
        fun provideCurrentData(): List<ItemImpressionModel>
    }

    @JvmStatic
    fun startTrackingForJavaClass(
        recyclerView: RecyclerView,
        currentDataProvider: CurrentDataProvider,
        searchId: String
    ) {
        trackRecyclerView(
            recyclerView,
            currentDataProvider::provideCurrentData,
            searchId
        )
    }

    @JvmStatic
    fun onCollectionUpdated(
        collectionViewKey: String,
        searchId: String,
        currentDataProvider: List<ItemImpressionModel>
    ) {

        currentDataProviderValue = currentDataProvider
        searchIdValue = searchId
        collectionViewId = collectionViewKey

        if (currentDataProvider[0].itemProperties.currency != InternalCurrencyCode.USD.name) {
            CFSetup().getUSDRate(
                currentDataProvider[0].itemProperties.currency,
                CfItemImpressionListener::getUSDRateAndLogRNEvent
            )
        } else {
            callCoreCollectionUpdated()
        }
    }

    @JvmStatic
    fun onCollectionUpdatedRN(
        collectionViewKey: String,
        searchId: String,
        currentDataProvider: String
    ) {

        val itemModels: Array<ItemImpressionModel> = Gson().fromJson(
            currentDataProvider,
            Array<ItemImpressionModel>::class.java
        )
        val itemList = java.util.ArrayList(itemModels.toMutableList())
        for (item in itemList) {
            try {
                when (item.itemProperties.type) {
                    ItemType.drug.name -> {
                        item.catalogProperties = item.catalogProperties.serializeToMap()
                            .toDataClass()
                    }
                    ItemType.blood.name -> {
                        item.catalogProperties = item.catalogProperties.serializeToMap()
                            .toDataClass()
                    }
                    ItemType.oxygen.name -> {
                        item.catalogProperties = item.catalogProperties.serializeToMap()
                            .toDataClass()
                    }
                    ItemType.medical_equipment.name -> {
                        item.catalogProperties = item.catalogProperties.serializeToMap()
                            .toDataClass()
                    }
                }
            } catch (ex: java.lang.Exception) {
                ExceptionManager.throwIllegalStateException("Invalid catalog object provided")
            }
            ECommerceConstants.isItemValueObjectValid(item.itemProperties, EComEventType.item)
        }

        currentDataProviderValue = itemList
        searchIdValue = searchId
        collectionViewId = collectionViewKey

        if (itemList.isNotEmpty()) {
            if (itemList[0].itemProperties.currency != InternalCurrencyCode.USD.name) {
                CFSetup().getUSDRate(
                    itemList[0].itemProperties.currency,
                    CfItemImpressionListener::getUSDRateAndLogRNEvent
                )
            } else {
                callCoreCollectionUpdated()
            }
        }
    }

    private fun getUSDRateAndLogRNEvent(usdRate: Float) {
        usdRateValue = usdRate
        callCoreCollectionUpdated()
    }

    // for react native SDK  to track elements in a coming from RN wrapper
    private fun callCoreCollectionUpdated() {
        BshCoreImpressionListener.onCollectionUpdated(null, collectionViewId,
            searchIdValue, currentDataProviderValue.map {
                RecyclerImpressionModel(
                    elementId = it.itemProperties.id,
                    contentBlock = ECommerceConstants.contentBlockName,
                    eventName = EComEventType.item.name,
                    catalogSubject = it.itemProperties.type,
                    itemProperties = ViewItemObject(
                        ItemAction.impression.name,
                        it.itemProperties, searchIdValue, usdRateValue, null
                    ),
                    catalogProperties = prepareCatalogObject(
                        it.itemProperties.id,
                        it.itemProperties.type,
                        it.catalogProperties
                    )
                )
            })
    }

}