package ai.causalfoundry.android.sdk.e_commerce.utils

import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.BloodCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.DrugCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.MedicalEquipmentCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.OxygenCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.internal_catalog_models.InternalBloodCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.internal_catalog_models.InternalDrugModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.internal_catalog_models.InternalMedicalEquipmentCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.internal_catalog_models.InternalOxygenCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.InternalCurrencyCode
import ai.causalfoundry.android.sdk.e_commerce.event_models.ItemModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.ItemTypeModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.meta_models.BloodMetaModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.meta_models.OxygenMetaModel
import ai.causalfoundry.android.sdk.e_commerce.event_types.EComEventType
import ai.causalfoundry.android.sdk.e_commerce.event_types.ItemType
import java.text.SimpleDateFormat
import java.util.*

/**
 * Created by Moiz Hassan on 22 March, 2023
 */


object ECommerceConstants {

    // SDK API endpoints
    const val contentBlockName: String = "e-commerce"

    fun getDateTime(milliSeconds: Long): String {
        // Create a DateFormatter object for displaying date in specified format.
        val sdf = SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSZZZZZ", Locale.ENGLISH)
        // Create a calendar object that will convert the date and time value in milliseconds to date.
        val calendar: Calendar = Calendar.getInstance()
        calendar.timeInMillis = milliSeconds
        return sdf.format(calendar.time)
    }

    fun isItemValueObjectValid(itemValue: ItemModel, eventType: EComEventType) {
        if (itemValue.id.isNullOrEmpty()) {
            ExceptionManager.throwIsRequiredException("item_id")
        } else if (itemValue.quantity < 0) {
            ExceptionManager.throwItemQuantityException()
        } else if (itemValue.price == null) {
            ExceptionManager.throwIsRequiredException("item_price")
        } else if (itemValue.price == -1f) {
            ExceptionManager.throwIsRequiredException("item_price")
        } else if (itemValue.currency.isNullOrEmpty()) {
            ExceptionManager.throwIsRequiredException("item_currency")
        } else if (!CoreConstants.enumContains<InternalCurrencyCode>(itemValue.currency)) {
            ExceptionManager.throwEnumException("CurrencyCode")
        } else if (itemValue.type.isNullOrEmpty()) {
            ExceptionManager.throwIsRequiredException("item_type")
        } else if (!CoreConstants.enumContains<ItemType>(itemValue.type)) {
            ExceptionManager.throwEnumException("ItemType")
        } else if (eventType == EComEventType.checkout && itemValue.type == ItemType.blood.name) {
            if (itemValue.meta == null) {
                ExceptionManager.throwIsRequiredException("Blood Meta Properties")
            } else if (itemValue.meta !is BloodMetaModel) {
                ExceptionManager.throwEnumException("Blood Meta Properties")
            }
        } else if (eventType == EComEventType.checkout && itemValue.type == ItemType.oxygen.name) {
            if (itemValue.meta == null) {
                ExceptionManager.throwIsRequiredException("Oxygen Meta Properties")
            } else if (itemValue.meta !is OxygenMetaModel) {
                ExceptionManager.throwEnumException("Oxygen Meta Properties")
            }
        }
    }

    fun isItemTypeObjectValid(itemValue: ItemTypeModel) {
        if (itemValue.item_id.isNullOrEmpty()) {
            ExceptionManager.throwIsRequiredException("item_id")
        } else if (itemValue.item_type.isNullOrEmpty()) {
            ExceptionManager.throwIsRequiredException("item_type")
        } else if (!CoreConstants.enumContains<ItemType>(itemValue.item_type)) {
            ExceptionManager.throwEnumException("ItemType")
        }
    }

    fun verifyCatalogForDrug(
        drugId: String,
        drugCatalogModel: DrugCatalogModel
    ): InternalDrugModel {
        when {
            drugId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException("drug Id")
            }
            drugCatalogModel.name.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException("name")
            }
        }

        return InternalDrugModel(
            id = drugId,
            name = CoreConstants.checkIfNull(drugCatalogModel.name),
            market_id = CoreConstants.checkIfNull(drugCatalogModel.market_id),
            description = CoreConstants.checkIfNull(drugCatalogModel.description),
            supplier_id = CoreConstants.checkIfNull(drugCatalogModel.supplier_id),
            supplier_name = CoreConstants.checkIfNull(drugCatalogModel.supplier_name),
            producer = CoreConstants.checkIfNull(drugCatalogModel.producer),
            packaging = CoreConstants.checkIfNull(drugCatalogModel.packaging),
            active_ingredients = drugCatalogModel.active_ingredients ?: arrayListOf(),
            drug_form = CoreConstants.checkIfNull(drugCatalogModel.drug_form),
            drug_strength = CoreConstants.checkIfNull(drugCatalogModel.drug_strength),
            atc_anatomical_group = CoreConstants.checkIfNull(drugCatalogModel.ATC_anatomical_group),
            otc_or_ethical = CoreConstants.checkIfNull(drugCatalogModel.OTC_or_ethical)
        )

    }

    fun verifyCatalogForBlood(
        itemId: String,
        bloodCatalogModel: BloodCatalogModel
    ): InternalBloodCatalogModel {

        when {
            itemId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException("blood item Id")
            }
            bloodCatalogModel.blood_group.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException("blood_group")
            }
        }

        return InternalBloodCatalogModel(
            id = itemId,
            market_id = CoreConstants.checkIfNull(bloodCatalogModel.market_id),
            blood_component = CoreConstants.checkIfNull(bloodCatalogModel.blood_component),
            blood_group = CoreConstants.checkIfNull(bloodCatalogModel.blood_group),
            packaging = CoreConstants.checkIfNull(bloodCatalogModel.packaging),
            packaging_size = bloodCatalogModel.packaging_size ?: 0f,
            packaging_units = CoreConstants.checkIfNull(bloodCatalogModel.packaging_units),
            supplier_id = CoreConstants.checkIfNull(bloodCatalogModel.supplier_id),
            supplier_name = CoreConstants.checkIfNull(bloodCatalogModel.supplier_name)
        )

    }

    fun verifyCatalogForOxygen(
        itemId: String,
        oxygenCatalogModel: OxygenCatalogModel
    ): InternalOxygenCatalogModel {

        when {
            itemId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException("oxygen item Id")
            }
        }
        return InternalOxygenCatalogModel(
            id = itemId,
            market_id = CoreConstants.checkIfNull(oxygenCatalogModel.market_id),
            packaging = CoreConstants.checkIfNull(oxygenCatalogModel.packaging),
            packaging_size = oxygenCatalogModel.packaging_size ?: 0f,
            packaging_units = CoreConstants.checkIfNull(oxygenCatalogModel.packaging_units),
            supplier_id = CoreConstants.checkIfNull(oxygenCatalogModel.supplier_id),
            supplier_name = CoreConstants.checkIfNull(oxygenCatalogModel.supplier_name)
        )

    }

    fun verifyCatalogForMedicalEquipment(
        itemId: String,
        medicalEquipmentCatalogModel: MedicalEquipmentCatalogModel
    ): InternalMedicalEquipmentCatalogModel {

        when {
            itemId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException("medical equipment item Id")
            }
            medicalEquipmentCatalogModel.name.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException("name")
            }
        }
        return InternalMedicalEquipmentCatalogModel(
            id = itemId,
            name = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.name),
            description = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.description),
            market_id = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.market_id),
            supplier_id = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.supplier_id),
            supplier_name = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.supplier_name),
            producer = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.producer),
            packaging = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.packaging),
            packaging_size = medicalEquipmentCatalogModel.packaging_size ?: 0f,
            packaging_units = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.packaging_units),
            category = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.category)
        )

    }

}
