package ai.causalfoundry.android.sdk.e_commerce.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.CancelCheckoutObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.CartObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.ItemTypeModel
import ai.causalfoundry.android.sdk.e_commerce.event_types.CancelType
import ai.causalfoundry.android.sdk.e_commerce.event_types.EComEventType
import ai.causalfoundry.android.sdk.e_commerce.utils.ECommerceConstants
import com.google.gson.Gson

/**
 * Created by Moiz Hassan on 22 March, 2023
 */

class CfLogCancelCheckoutEvent {

    /**
     * CfLogCancelCheckoutEvent is used to log the event when the order/cart is canceled.
     */

    data class Builder(
        internal var checkout_id: String? = null,
        internal var type: String? = null,
        internal var item_list: ArrayList<ItemTypeModel> = arrayListOf(),
        internal var reason: String = "",
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        private lateinit var cartObject: CartObject

        /**
         * setCheckoutId can be used to log the orderId the event is logged for. if the order ID is
         * not there then it is recommended to include the unique cartId for the cart items so
         * that they can be tracked. In case the cart Id is also not there, you can use the userId
         * as a unique element for the order.
         */
        fun setCheckoutId(checkout_id: String) = apply { this.checkout_id = checkout_id }

        /**
         * setCancelType can be used to log the type of the checkout that is being cancelled.
         * It can be order or card but should reflect the correct type in correspondence to
         * the id provided.
         */
        fun setCancelType(cancelType: CancelType) = apply { this.type = cancelType.name }
        fun setCancelType(cancelType: String) = apply {
            if (CoreConstants.enumContains<CancelType>(cancelType)) {
                this.type = cancelType
            } else {
                ExceptionManager.throwEnumException("CancelType")
            }
        }

        /**
         * setOrderItemTypes is required to provide the list of item types that are present
         * in the order being cancelled
         */

        /**
         * addItem can be used to add the item being orders in to the checkout list. This log can
         * be used to add one item to the log at a time. Order item should be in a valid format.
         * With elements of the order as:
         * ItemTypeModel(itemID, type) You can add multiple addOrder functions to one event to include
         * all the items in the order.
         */
        fun addItem(itemModel: ItemTypeModel) = apply {
            ECommerceConstants.isItemTypeObjectValid(itemModel)
            this.item_list.add(itemModel)
        }

        /**
         * addItem can be used to pass the whole item as a Json String object as well. You can use
         * the POJO ItemTypeModel to parse the data int he required format and pass that to this
         * function as a string to log the event. You can use Gson to convert the object to string
         * but SDK will parse the Json string back to POJO so pass it in the log. This method
         * should be used with caution and is suitable for react native bridge.
         */
        fun addItem(itemJsonString: String) = apply {
            val item: ItemTypeModel = Gson().fromJson(itemJsonString, ItemTypeModel::class.java)
            ECommerceConstants.isItemTypeObjectValid(item)
            this.item_list.add(item)
        }

        /**
         * addItemList can be used to add the whole list to the log at once, the format should be
         * ArrayList<ItemTypeModel> to log the event successfully. Order item should be in a valid format.
         * With elements of the orderObject as:
         * ItemTypeModel(itemID, type) You should use only one addItemList with event or else the list
         * will only save the later one.
         */
        fun addItemList(itemList: ArrayList<ItemTypeModel>) = apply {
            this.item_list.clear()
            for (item in itemList) {
                ECommerceConstants.isItemTypeObjectValid(item)
            }
            this.item_list.addAll(itemList)
        }

        /**
         * addItemList can be used to add the whole list to the log at once, the format should be
         * ArrayList<ItemTypeModel> to log the event successfully. But the input param is of type
         * string , this is special use case for react native logs where list can be passed as
         * Json string and can be used to log the event. Order item should be in a valid format.
         * With elements of the orderObject as:
         * ItemTypeModel(itemID, type) You should use only one addItemList with event or else the list
         * will only save the later one.
         */
        fun addItemList(itemListString: String) = apply {
            this.item_list.clear()
            val itemModels: Array<ItemTypeModel> = Gson().fromJson(
                itemListString,
                Array<ItemTypeModel>::class.java
            )
            val itemList = java.util.ArrayList(itemModels.toMutableList())
            for (item in itemList) {
                ECommerceConstants.isItemTypeObjectValid(item)
            }
            this.item_list.addAll(itemList)
        }

        /**
         * setCancelReason is required to define the reason for which the item is being cancelled.
         */
        fun setCancelReason(reason: String) = apply {
            this.reason = reason
        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if id provided is null or not
                 * provided at all.
                 */
                checkout_id.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("id")
                }

                /**
                 * Will throw an exception for the developer if type provided is null or not
                 * provided at all.
                 */
                type.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("cancel_type")
                }

                /**
                 * Will throw an exception for the developer if item_types provided is null or not
                 * provided at all.
                 */
                item_list.isEmpty() -> {
                    ExceptionManager.throwIsRequiredException("item_list")
                }

//                /**
//                 * Will throw an exception for the developer if reason provided is null or not
//                 * provided at all.
//                 */
//                reason.isNullOrEmpty() -> {
//                    ExceptionManager.throwIsRequiredException("reason")
//                }

                else -> {
                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val cancelCheckoutObject = CancelCheckoutObject(
                        checkout_id!!, type!!,
                        item_list.distinctBy { it.item_id }, reason, meta
                    )
                    CFSetup().track(
                        ECommerceConstants.contentBlockName, EComEventType.cancel_checkout.name,
                        cancelCheckoutObject, update_immediately
                    )
                }
            }
        }

    }
}
