package ai.causalfoundry.android.sdk.e_commerce.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.ItemReportObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.ItemTypeModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.ReportObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.StoreObject
import ai.causalfoundry.android.sdk.e_commerce.event_types.EComEventType
import ai.causalfoundry.android.sdk.e_commerce.event_types.ItemType
import ai.causalfoundry.android.sdk.e_commerce.utils.ECommerceConstants
import com.google.gson.Gson

/**
 * Created by Moiz Hassan on 03 April, 2023
 */

class CfLogItemReportEvent {

    /**
     * CfLogItemReportEvent is used to log the event an item is reported.
     */

    data class Builder(
        internal var item_object: ItemTypeModel? = null,
        internal var store_object: StoreObject? = null,
        internal var report_object: ReportObject? = null,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * setItem is for the providing item Id and type for the item in question.
         * The object should be based on the ItemTypeModel or a string that can be
         * converted to the object with proper param names. in-case the names are not correct
         * the SDK will throw an exception. Below is the function for providing item as a string.
         */
        fun setItem(item_object: ItemTypeModel) = apply {
            if (CoreConstants.enumContains<ItemType>(item_object.item_type)) {
                this.item_object = item_object
            } else {
                ExceptionManager.throwEnumException(ItemType::class.java.simpleName)
            }
        }

        /**
         * setItem is for the providing item Id and type for the item in question.
         * The object should be based on the ItemTypeModel or a string that can be
         * converted to the object with proper param names. in-case the names are not correct
         * the SDK will throw an exception. Below is the function for providing item as a string.
         */
        fun setItem(item_object: String) = apply {
            this.item_object = Gson().fromJson(item_object, ItemTypeModel::class.java)
        }

        /**
         * setStoreObject is for the providing store object details for the item report if any.
         * The object should be based on the StoreObject or a string that can be
         * converted to the object with proper param names. in-case the names are not correct
         * the SDK will throw an exception. Below is the function for providing item as a string.
         */
        fun setStoreObject(store_object: StoreObject) = apply {
            this.store_object = store_object
        }
        fun setStoreObject(store_object: String) = apply {
            this.store_object = Gson().fromJson(store_object, StoreObject::class.java)

        }

        /**
         * setReportObject is for the providing report object details for the item report if any.
         * The object should be based on the ReportObject or a string that can be
         * converted to the object with proper param names. In-case the names are not correct
         * the SDK will throw an exception. Below is the function for providing item as a string.
         */
        fun setReportObject(report_object: ReportObject) = apply {
            this.report_object = report_object
        }
        fun setReportObject(report_object: String) = apply {
            this.report_object = Gson().fromJson(report_object, ReportObject::class.java)
        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if id provided is null or not
                 * provided at all.
                 */
                item_object == null -> {
                    ExceptionManager.throwIsRequiredException("item_object")
                }
                item_object!!.item_id.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("item_id")
                }

                /**
                 * Will throw an exception for the developer if type provided is null or not
                 * provided at all.
                 */
                item_object!!.item_type.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("item_type")
                }

                /**
                 * Will throw an exception for the developer if store_object provided is null or not
                 * provided at all.
                 */
                store_object == null -> {
                    ExceptionManager.throwIsRequiredException("store_object")
                }
                store_object!!.id.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException("store_id")
                }

                /**
                 * Will throw an exception for the developer if report_object provided is null or not
                 * provided at all.
                 */
                report_object == null -> {
                    ExceptionManager.throwIsRequiredException("report_object")
                }
                report_object!!.id.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("report_id")
                }
                report_object!!.short_desc.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("report_short_desc")
                }

                else -> {
                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val itemReportObject = ItemReportObject(
                        item_object!!, store_object!!, report_object!!, meta
                    )
                    CFSetup().track(
                        ECommerceConstants.contentBlockName, EComEventType.item_report.name,
                        itemReportObject, update_immediately
                    )
                }
            }
        }

    }
}
