package ai.causalfoundry.android.sdk.e_commerce.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.ItemVerificationObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.ItemInfoObject
import ai.causalfoundry.android.sdk.e_commerce.event_types.EComEventType
import ai.causalfoundry.android.sdk.e_commerce.event_types.ItemType
import ai.causalfoundry.android.sdk.e_commerce.event_types.ScanChannel
import ai.causalfoundry.android.sdk.e_commerce.event_types.ScanType
import ai.causalfoundry.android.sdk.e_commerce.utils.ECommerceConstants
import com.google.gson.Gson

/**
 * Created by Moiz Hassan on 04 April, 2023
 */

class CfLogItemVerificationEvent {

    /**
     * CfLogItemVerificationEvent is used to log the event an item is verified/scanned by the user.
     */

    data class Builder(
        internal var scan_channel: String? = null,
        internal var scan_type: String? = null,
        internal var is_successful: Boolean? = null,
        internal var item_info: ItemInfoObject? = null,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * setScanChannel is for the providing scan channel used for verification.
         */
        fun setScanChannel(scan_channel: ScanChannel) = apply {
            this.scan_channel = scan_channel.name
        }

        fun setScanChannel(scan_channel: String) = apply {
            if (CoreConstants.enumContains<ScanChannel>(scan_channel)) {
                this.scan_channel = scan_channel
            } else {
                ExceptionManager.throwEnumException(
                    EComEventType.item_verification.name,
                    ScanChannel::class.java.simpleName
                )
            }
        }

        /**
         * setScanType is for the providing scan Type used for verification.
         */
        fun setScanType(scan_type: ScanType) = apply {
            this.scan_type = scan_type.name
        }

        fun setScanType(scan_type: String) = apply {
            if (CoreConstants.enumContains<ScanType>(scan_type)) {
                this.scan_type = scan_type
            } else {
                ExceptionManager.throwEnumException(
                    EComEventType.item_verification.name,
                    ScanType::class.java.simpleName
                )
            }
        }

        /**
         * isSuccessful is for the providing if verification was successful or not.
         */
        fun isSuccessful(is_successful: Boolean) = apply {
            this.is_successful = is_successful
        }

        /**
         * setItemInfo is for the providing item info details returned by the verification.
         * The object should be based on the ItemInfoObject or a string that can be
         * converted to the object with proper param names. in-case the names are not correct
         * the SDK will throw an exception. Below is the function for providing item as a string.
         */
        fun setItemInfo(item_info: ItemInfoObject) = apply {
            if (CoreConstants.enumContains<ItemType>(item_info.type)) {
                this.item_info = item_info
            } else {
                ExceptionManager.throwEnumException(
                    EComEventType.item_verification.name,
                    ItemType::class.java.simpleName
                )
            }
        }

        /**
         * setItemInfo is for the providing item info details returned by the verification.
         * The object should be based on the ItemInfoObject or a string that can be
         * converted to the object with proper param names. in-case the names are not correct
         * the SDK will throw an exception. Below is the function for providing item as a string.
         */
        fun setItemInfo(item_info: String) = apply {
            this.item_info = Gson().fromJson(item_info, ItemInfoObject::class.java)

        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if scan_channel provided is null
                 * or not provided at all.
                 */
                scan_channel.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.item_verification.name,
                        "scan_channel"
                    )
                }

                /**
                 * Will throw an exception for the developer if scan_type provided is null or not
                 * provided at all.
                 */
                scan_type.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.item_verification.name,
                        "scan_type"
                    )
                }

                /**
                 * Will throw an exception for the developer if is_successful provided is null or not
                 * provided at all.
                 */
                is_successful == null -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.item_verification.name,
                        "is_successful"
                    )
                }

                else -> {

                    if (is_successful!! && item_info != null) {
                        /**
                         * Will throw an exception for the developer if item id provided is null or not
                         * provided at all.
                         */
                        if (item_info?.id.isNullOrEmpty()) {
                            ExceptionManager.throwIsRequiredException(
                                EComEventType.item_verification.name,
                                "item_id"
                            )
                        }

                        /**
                         * Will throw an exception for the developer if item type provided is null or not
                         * provided at all.
                         */
                        if (item_info?.type.isNullOrEmpty()) {
                            ExceptionManager.throwIsRequiredException(
                                EComEventType.item_verification.name,
                                "item_type"
                            )
                        }
                    } else if (is_successful!! && item_info == null) {
                        ExceptionManager.throwIsRequiredException(
                            EComEventType.item_verification.name,
                            "item_info"
                        )
                    } else {
                        item_info = null
                    }


                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val itemVerificationObject = ItemVerificationObject(
                        scan_channel!!, scan_type!!, is_successful!!, item_info, meta
                    )
                    CFSetup().track(
                        ECommerceConstants.contentBlockName, EComEventType.item_verification.name,
                        itemVerificationObject, update_immediately
                    )
                }
            }
        }

    }
}
