package ai.causalfoundry.android.sdk.e_commerce.utils

import ai.causalfoundry.android.sdk.core.catalog.catalog_models.CatalogSubject
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.BloodCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.DrugCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.FacilityCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.GroceryCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.MedicalEquipmentCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.OxygenCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.internal_catalog_models.InternalBloodCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.internal_catalog_models.InternalDrugModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.internal_catalog_models.InternalFacilityCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.internal_catalog_models.InternalGroceryCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.internal_catalog_models.InternalMedicalEquipmentCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.internal_catalog_models.InternalOxygenCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.ItemModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.ItemTypeModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.meta_models.BloodMetaModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.meta_models.OxygenMetaModel
import ai.causalfoundry.android.sdk.e_commerce.event_types.EComEventType
import ai.causalfoundry.android.sdk.e_commerce.event_types.InternalCurrencyCode
import ai.causalfoundry.android.sdk.e_commerce.event_types.ItemType

/**
 * Created by Moiz Hassan on 22 March, 2023
 */


object ECommerceConstants {

    // SDK API endpoints
    const val contentBlockName: String = "e-commerce"

    fun isItemValueObjectValid(itemValue: ItemModel, eventType: EComEventType) {

        val eventName = eventType.name

        if (itemValue.id.isNullOrEmpty()) {
            ExceptionManager.throwIsRequiredException(eventName, "item_id")
        } else if (itemValue.quantity < 0) {
            ExceptionManager.throwItemQuantityException(eventName)
        } else if (itemValue.price == null) {
            ExceptionManager.throwIsRequiredException(eventName, "item_price")
        } else if (itemValue.price == -1f) {
            ExceptionManager.throwIsRequiredException(eventName, "item_price")
        } else if (itemValue.currency.isNullOrEmpty()) {
            ExceptionManager.throwIsRequiredException(eventName, "item_currency")
        } else if (!CoreConstants.enumContains<InternalCurrencyCode>(itemValue.currency)) {
            ExceptionManager.throwEnumException(eventName, "CurrencyCode")
        } else if (itemValue.type.isNullOrEmpty()) {
            ExceptionManager.throwIsRequiredException(eventName, "item_type")
        } else if (!CoreConstants.enumContains<ItemType>(itemValue.type)) {
            ExceptionManager.throwEnumException(eventName, "ItemType")
        } else if (eventType == EComEventType.checkout && itemValue.type == ItemType.blood.name) {
            if (itemValue.meta == null) {
                ExceptionManager.throwIsRequiredException(eventName, "Blood Meta Properties")
            } else if (itemValue.meta !is BloodMetaModel) {
                ExceptionManager.throwEnumException(eventName, "Blood Meta Properties")
            }
        } else if (eventType == EComEventType.checkout && itemValue.type == ItemType.oxygen.name) {
            if (itemValue.meta == null) {
                ExceptionManager.throwIsRequiredException(eventName, "Oxygen Meta Properties")
            } else if (itemValue.meta !is OxygenMetaModel) {
                ExceptionManager.throwEnumException(eventName, "Oxygen Meta Properties")
            }
        }
    }

    fun isItemTypeObjectValid(itemValue: ItemTypeModel, eventType: EComEventType) {

        val eventName = eventType.name

        if (itemValue.item_id.isNullOrEmpty()) {
            ExceptionManager.throwIsRequiredException(eventName, "item_id")
        } else if (itemValue.item_type.isNullOrEmpty()) {
            ExceptionManager.throwIsRequiredException(eventName, "item_type")
        } else if (!CoreConstants.enumContains<ItemType>(itemValue.item_type)) {
            ExceptionManager.throwEnumException(eventName, "ItemType")
        }
    }

    fun verifyCatalogForDrug(
        drugId: String,
        drugCatalogModel: DrugCatalogModel
    ): InternalDrugModel {

        val catalogName = CatalogSubject.drug.name + " catalog"

        when {
            drugId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "drug Id")
            }

            drugCatalogModel.name.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "name")
            }
        }

        return InternalDrugModel(
            id = drugId,
            name = CoreConstants.checkIfNull(drugCatalogModel.name),
            market_id = CoreConstants.checkIfNull(drugCatalogModel.market_id),
            description = CoreConstants.checkIfNull(drugCatalogModel.description),
            supplier_id = CoreConstants.checkIfNull(drugCatalogModel.supplier_id),
            supplier_name = CoreConstants.checkIfNull(drugCatalogModel.supplier_name),
            producer = CoreConstants.checkIfNull(drugCatalogModel.producer),
            packaging = CoreConstants.checkIfNull(drugCatalogModel.packaging),
            active_ingredients = drugCatalogModel.active_ingredients ?: arrayListOf(),
            drug_form = CoreConstants.checkIfNull(drugCatalogModel.drug_form),
            drug_strength = CoreConstants.checkIfNull(drugCatalogModel.drug_strength),
            atc_anatomical_group = CoreConstants.checkIfNull(drugCatalogModel.ATC_anatomical_group),
            otc_or_ethical = CoreConstants.checkIfNull(drugCatalogModel.OTC_or_ethical)
        )

    }

    fun verifyCatalogForGrocery(
        itemId: String,
        groceryCatalogModel: GroceryCatalogModel
    ): InternalGroceryCatalogModel {

        val catalogName = CatalogSubject.grocery.name + " catalog"

        when {
            itemId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "grocery item Id")
            }

            groceryCatalogModel.name.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "name")
            }
        }

        return InternalGroceryCatalogModel(
            id = itemId,
            name = CoreConstants.checkIfNull(groceryCatalogModel.name),
            category = CoreConstants.checkIfNull(groceryCatalogModel.category),
            market_id = CoreConstants.checkIfNull(groceryCatalogModel.market_id),
            description = CoreConstants.checkIfNull(groceryCatalogModel.description),
            supplier_id = CoreConstants.checkIfNull(groceryCatalogModel.supplier_id),
            supplier_name = CoreConstants.checkIfNull(groceryCatalogModel.supplier_name),
            producer = CoreConstants.checkIfNull(groceryCatalogModel.producer),
            packaging = CoreConstants.checkIfNull(groceryCatalogModel.packaging),
            packaging_size = groceryCatalogModel.packaging_size ?: 0f,
            packaging_units = CoreConstants.checkIfNull(groceryCatalogModel.packaging_units),
            active_ingredients = groceryCatalogModel.active_ingredients ?: arrayListOf()
        )

    }

    fun verifyCatalogForBlood(
        itemId: String,
        bloodCatalogModel: BloodCatalogModel
    ): InternalBloodCatalogModel {

        val catalogName = CatalogSubject.blood.name + " catalog"

        when {
            itemId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "blood item Id")
            }

            bloodCatalogModel.blood_group.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "blood_group")
            }
        }

        return InternalBloodCatalogModel(
            id = itemId,
            market_id = CoreConstants.checkIfNull(bloodCatalogModel.market_id),
            blood_component = CoreConstants.checkIfNull(bloodCatalogModel.blood_component),
            blood_group = CoreConstants.checkIfNull(bloodCatalogModel.blood_group),
            packaging = CoreConstants.checkIfNull(bloodCatalogModel.packaging),
            packaging_size = bloodCatalogModel.packaging_size ?: 0f,
            packaging_units = CoreConstants.checkIfNull(bloodCatalogModel.packaging_units),
            supplier_id = CoreConstants.checkIfNull(bloodCatalogModel.supplier_id),
            supplier_name = CoreConstants.checkIfNull(bloodCatalogModel.supplier_name)
        )

    }

    fun verifyCatalogForOxygen(
        itemId: String,
        oxygenCatalogModel: OxygenCatalogModel
    ): InternalOxygenCatalogModel {

        val catalogName = CatalogSubject.oxygen.name + " catalog"

        when {
            itemId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "oxygen item Id")
            }
        }
        return InternalOxygenCatalogModel(
            id = itemId,
            market_id = CoreConstants.checkIfNull(oxygenCatalogModel.market_id),
            packaging = CoreConstants.checkIfNull(oxygenCatalogModel.packaging),
            packaging_size = oxygenCatalogModel.packaging_size ?: 0f,
            packaging_units = CoreConstants.checkIfNull(oxygenCatalogModel.packaging_units),
            supplier_id = CoreConstants.checkIfNull(oxygenCatalogModel.supplier_id),
            supplier_name = CoreConstants.checkIfNull(oxygenCatalogModel.supplier_name)
        )

    }

    fun verifyCatalogForMedicalEquipment(
        itemId: String,
        medicalEquipmentCatalogModel: MedicalEquipmentCatalogModel
    ): InternalMedicalEquipmentCatalogModel {

        val catalogName = CatalogSubject.medical_equipment.name + " catalog"

        when {
            itemId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "medical equipment item Id")
            }

            medicalEquipmentCatalogModel.name.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "name")
            }
        }
        return InternalMedicalEquipmentCatalogModel(
            id = itemId,
            name = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.name),
            description = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.description),
            market_id = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.market_id),
            supplier_id = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.supplier_id),
            supplier_name = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.supplier_name),
            producer = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.producer),
            packaging = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.packaging),
            packaging_size = medicalEquipmentCatalogModel.packaging_size ?: 0f,
            packaging_units = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.packaging_units),
            category = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.category)
        )

    }

    fun verifyCatalogForFacility(
        facilityId: String,
        facilityCatalogModel: FacilityCatalogModel
    ): InternalFacilityCatalogModel {

        val catalogName = CatalogSubject.facility.name + " catalog"

        when {
            facilityId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "facility Id")
            }

            facilityCatalogModel.name.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "name")
            }

            facilityCatalogModel.type.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "type")
            }
        }
        return InternalFacilityCatalogModel(
            id = facilityId,
            name = CoreConstants.checkIfNull(facilityCatalogModel.name),
            type = CoreConstants.checkIfNull(facilityCatalogModel.type),
            country = CoreConstants.checkIfNull(facilityCatalogModel.country),
            region_state = CoreConstants.checkIfNull(facilityCatalogModel.region_state),
            city = CoreConstants.checkIfNull(facilityCatalogModel.city),
            is_active = facilityCatalogModel.is_active ?: false,
            has_delivery = facilityCatalogModel.has_delivery ?: false,
            is_sponsored = facilityCatalogModel.is_sponsored ?: false
        )

    }

}
