package ai.causalfoundry.android.sdk.e_commerce.impression_listerner

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.impression_listener.CfCoreImpressionListener
import ai.causalfoundry.android.sdk.core.impression_listener.impression_model.RecyclerImpressionModel
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.core.utils.serializeToMap
import ai.causalfoundry.android.sdk.core.utils.toDataClass
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.BloodCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.DrugCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.FacilityCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.GroceryCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.MedicalEquipmentCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.OxygenCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.ViewItemObject
import ai.causalfoundry.android.sdk.e_commerce.event_types.EComEventType
import ai.causalfoundry.android.sdk.e_commerce.event_types.InternalCurrencyCode
import ai.causalfoundry.android.sdk.e_commerce.event_types.ItemAction
import ai.causalfoundry.android.sdk.e_commerce.event_types.ItemType
import ai.causalfoundry.android.sdk.e_commerce.impression_listerner.event_model.ItemImpressionModel
import ai.causalfoundry.android.sdk.e_commerce.utils.ECommerceConstants
import androidx.recyclerview.widget.RecyclerView
import com.google.gson.Gson

/**
 * Created by Moiz Hassan on 22 March, 2023
 */


object CfItemImpressionListener {

    private var usdRateValue: Float = 1f
    private lateinit var recyclerViewValue: RecyclerView
    private lateinit var currentDataProviderValue: List<ItemImpressionModel>
    private var searchIdValue: String = ""
    private var collectionViewId: String = ""

    @JvmStatic
    fun trackRecyclerView(
        recyclerView: RecyclerView,
        currentDataProvider: () -> List<ItemImpressionModel>,
        searchId: String
    ) {

        recyclerViewValue = recyclerView
        currentDataProviderValue = currentDataProvider()
        searchIdValue = searchId

        if (currentDataProvider()[0].item_properties.currency != InternalCurrencyCode.USD.name) {
            CFSetup().getUSDRate(
                currentDataProvider()[0].item_properties.currency,
                CfItemImpressionListener::getUSDRateAndLogEvent
            )
        } else {
            callCoreImpressionListenerTrackRecyclerView()
        }

    }

    // for native SDK  to track elements in a recyclerview
    private fun callCoreImpressionListenerTrackRecyclerView() {
        CfCoreImpressionListener.trackRecyclerView(recyclerViewValue, currentDataProvider = {
            currentDataProviderValue.map {
                RecyclerImpressionModel(
                    element_id = it.item_properties.id,
                    content_block = ECommerceConstants.contentBlockName,
                    event_name = EComEventType.item.name,
                    catalog_subject = it.item_properties.type,
                    item_properties = ViewItemObject(
                        ItemAction.impression.name,
                        it.item_properties, searchIdValue, usdRateValue, null
                    ),
                    catalog_properties = prepareCatalogObject(
                        it.item_properties.id,
                        it.item_properties.type,
                        it.catalog_properties
                    )
                )
            }
        }, searchIdValue, null)
    }

    private fun getUSDRateAndLogEvent(usdRate: Float) {
        usdRateValue = usdRate
        callCoreImpressionListenerTrackRecyclerView()
    }

    private fun prepareCatalogObject(
        itemId: String,
        itemType: String,
        catalogModel: Any?
    ): Any? {
        return when (itemType) {
            ItemType.drug.name -> {
                ECommerceConstants.verifyCatalogForDrug(itemId, catalogModel as DrugCatalogModel)
            }

            ItemType.grocery.name -> {
                ECommerceConstants.verifyCatalogForGrocery(
                    itemId,
                    catalogModel as GroceryCatalogModel
                )
            }

            ItemType.facility.name -> {
                ECommerceConstants.verifyCatalogForFacility(
                    itemId,
                    catalogModel as FacilityCatalogModel
                )
            }

            ItemType.blood.name -> {
                ECommerceConstants.verifyCatalogForBlood(
                    itemId,
                    catalogModel as BloodCatalogModel
                )
            }

            ItemType.oxygen.name -> {
                ECommerceConstants.verifyCatalogForOxygen(
                    itemId,
                    catalogModel as OxygenCatalogModel
                )
            }

            ItemType.medical_equipment.name -> {
                ECommerceConstants.verifyCatalogForMedicalEquipment(
                    itemId,
                    catalogModel as MedicalEquipmentCatalogModel
                )
            }

            else -> {
                null
            }
        }

    }

    interface CurrentDataProvider {
        fun provideCurrentData(): List<ItemImpressionModel>
    }

    @JvmStatic
    fun startTrackingForJavaClass(
        recyclerView: RecyclerView,
        currentDataProvider: CurrentDataProvider,
        searchId: String
    ) {
        trackRecyclerView(
            recyclerView,
            currentDataProvider::provideCurrentData,
            searchId
        )
    }

    @JvmStatic
    fun onCollectionUpdated(
        collectionViewKey: String,
        searchId: String,
        currentDataProvider: List<ItemImpressionModel>
    ) {

        currentDataProviderValue = currentDataProvider
        searchIdValue = searchId
        collectionViewId = collectionViewKey

        if (currentDataProvider[0].item_properties.currency != InternalCurrencyCode.USD.name) {
            CFSetup().getUSDRate(
                currentDataProvider[0].item_properties.currency,
                CfItemImpressionListener::getUSDRateAndLogRNEvent
            )
        } else {
            callCoreCollectionUpdated()
        }
    }

    @JvmStatic
    fun onCollectionUpdatedRN(
        collectionViewKey: String,
        searchId: String,
        currentDataProvider: String
    ) {

        val itemModels: Array<ItemImpressionModel> = Gson().fromJson(
            currentDataProvider,
            Array<ItemImpressionModel>::class.java
        )

        val itemList = java.util.ArrayList(itemModels.toMutableList())
        for (item in itemList) {
            try {
                when (item.item_properties.type) {
                    ItemType.drug.name -> {
                        item.catalog_properties = item.catalog_properties.serializeToMap()
                            .toDataClass() as DrugCatalogModel
                    }

                    ItemType.grocery.name -> {
                        item.catalog_properties = item.catalog_properties.serializeToMap()
                            .toDataClass() as GroceryCatalogModel
                    }

                    ItemType.facility.name -> {
                        item.catalog_properties = item.catalog_properties.serializeToMap()
                            .toDataClass() as FacilityCatalogModel
                    }

                    ItemType.blood.name -> {
                        item.catalog_properties = item.catalog_properties.serializeToMap()
                            .toDataClass() as BloodCatalogModel
                    }

                    ItemType.oxygen.name -> {
                        item.catalog_properties = item.catalog_properties.serializeToMap()
                            .toDataClass() as OxygenCatalogModel
                    }

                    ItemType.medical_equipment.name -> {
                        item.catalog_properties = item.catalog_properties.serializeToMap()
                            .toDataClass() as MedicalEquipmentCatalogModel
                    }
                }
            } catch (ex: java.lang.Exception) {
                ExceptionManager.throwIllegalStateException(
                    "impression_listener",
                    "Invalid catalog object provided"
                )
            }
            ECommerceConstants.isItemValueObjectValid(item.item_properties, EComEventType.item)
        }

        currentDataProviderValue = itemList
        searchIdValue = searchId
        collectionViewId = collectionViewKey

        if (itemList.isNotEmpty()) {
            if (itemList[0].item_properties.currency != InternalCurrencyCode.USD.name) {
                CFSetup().getUSDRate(
                    itemList[0].item_properties.currency,
                    CfItemImpressionListener::getUSDRateAndLogRNEvent
                )
            } else {
                callCoreCollectionUpdated()
            }
        }
    }

    private fun getUSDRateAndLogRNEvent(usdRate: Float) {
        usdRateValue = usdRate
        callCoreCollectionUpdated()
    }

    // for react native SDK  to track elements in a coming from RN wrapper
    private fun callCoreCollectionUpdated() {
        CfCoreImpressionListener.onCollectionUpdated(null, collectionViewId,
            searchIdValue, currentDataProviderValue.map {
                RecyclerImpressionModel(
                    element_id = it.item_properties.id,
                    content_block = ECommerceConstants.contentBlockName,
                    event_name = EComEventType.item.name,
                    catalog_subject = it.item_properties.type,
                    item_properties = ViewItemObject(
                        ItemAction.impression.name,
                        it.item_properties, searchIdValue, usdRateValue, null
                    ),
                    catalog_properties = prepareCatalogObject(
                        it.item_properties.id,
                        it.item_properties.type,
                        it.catalog_properties
                    )
                )
            })
    }

}