package ai.causalfoundry.android.sdk.e_commerce.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.CartObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.ItemModel
import ai.causalfoundry.android.sdk.e_commerce.event_types.CartAction
import ai.causalfoundry.android.sdk.e_commerce.event_types.EComEventType
import ai.causalfoundry.android.sdk.e_commerce.event_types.InternalCurrencyCode
import ai.causalfoundry.android.sdk.e_commerce.utils.ECommerceConstants
import com.google.gson.Gson
import kotlin.math.roundToInt

/**
 * Created by Moiz Hassan on 22 March, 2023
 */

class CfLogCartEvent {

    /**
     * CfLogCartEvent is used to log the events related to cart. You can use this event to log
     * when an item was added or removed form the cart.
     */

    data class Builder(
        internal var cartIdValue: String? = null,
        internal var cartActionValue: String? = null,
        internal var itemValue: ItemModel = ItemModel("", 1, -1f, "", "", 0f, "", "", "", null),
        internal var cartPriceValue: Float? = null,
        internal var currencyValue: String? = null,
        private var meta: Any? = null,
        private var isUpdateImmediately: Boolean = CoreConstants.updateImmediately
    ) {

        private lateinit var cartObject: CartObject

        /**
         * setCartId can be used to log the cartId the event is logged for. It is recommended to
         * include the unique cartId for the cart items so that they can be tracked.
         */
        fun setCartId(cartId: String) = apply { this.cartIdValue = cartId }

        /**
         * setCartAction is required to pass the actions for cart the logged is triggered on. By
         * default the SDK provides 2 main list actions for e-commerce apps. Which includes
         * addItem and removeItem
         * setCartAction provides 2 approaches for logging list events, one is with enums and the
         * other is with string. Below is the function to log cartAction event using enum type.
         */
        fun setCartAction(cartAction: CartAction) = apply { this.cartActionValue = cartAction.name }

        /**
         * setCartAction is required to pass the actions for cart the logged is triggered on. By
         * default the SDK provides 2 main list actions for e-commerce apps. Which includes
         * addItem and removeItem.
         * setCartAction provides 2 approaches for logging list events, one is with enums and the
         * other is with string. Below is the function to log listAction event using string type.
         * Remember to note that with string type, you need to pass the values as provided
         * in the enum or else the events will be discarded
         */
        fun setCartAction(cartAction: String) = apply {
            if (CoreConstants.enumContains<CartAction>(cartAction)) {
                this.cartActionValue = cartAction
            } else {
                ExceptionManager.throwEnumException(
                    EComEventType.cart.name,
                    CartAction::class.java.simpleName
                )
            }
        }

        /**
         * setItem can be used to pass the whole item as an object as well. You can use the POJO
         * ItemModel to parse the data int he required format and pass that to this function to
         * log the event.
         */
        fun setItem(item: ItemModel) = apply {
            this.itemValue = item
        }

        /**
         * setItem can be used to pass the whole item as a Json String object as well. You can use
         * the POJO ItemModel to parse the data int he required format and pass that to this
         * function as a string to log the event. You can use Gson to convert the object to string
         * but SDK will parse the Json string back to POJO so pass it in the log. This method
         * should be used with caution and is suitable for react native bridge.
         */
        fun setItem(item: String) = apply {
            val itemObject: ItemModel = Gson().fromJson(item, ItemModel::class.java)
            this.itemValue = itemObject
        }

        /**
         * setCartPrice is required to log the total price for cart being logged. Details
         * about the cart are to be provided in the catalog.
         */
        fun setCartPrice(cartPrice: Int) = apply { this.cartPriceValue = cartPrice.toFloat() }
        fun setCartPrice(cartPrice: Float) = apply {
            this.cartPriceValue = ((cartPrice * 100.0).roundToInt() / 100.0).toFloat()
        }

        fun setCartPrice(cartPrice: Double) = apply {
            this.cartPriceValue = ((cartPrice * 100.0).roundToInt() / 100.0).toFloat()
        }

        /**
         * setCurrency is required to log the currency for cart price being logged. Details
         * about the cart are to be provided in the catalog.
         */
//        fun setCurrency(currency_code: CurrencyCode) = apply { this.currency_value = currency_code.name }
        fun setCurrency(currencyCode: String) = apply {
            if (CoreConstants.enumContains<InternalCurrencyCode>(currencyCode)) {
                this.currencyValue = currencyCode
            } else {
                ExceptionManager.throwEnumException(EComEventType.cart.name, "CurrencyCode")
            }
        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(updateImmediately: Boolean) =
            apply { this.isUpdateImmediately = updateImmediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if cart_id provided is null or not
                 * provided at all.
                 */
                cartIdValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(EComEventType.cart.name, "cart_id")
                }

                /**
                 * Will throw an exception for the developer if cart_action provided is null or not
                 * provided at all.
                 */
                cartActionValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.cart.name,
                        CartAction::class.java.simpleName
                    )
                }

                /**
                 * Will throw an exception for the developer if cart_price provided is null or not
                 * provided at all.
                 */
                cartPriceValue == null -> {
                    ExceptionManager.throwIsRequiredException(EComEventType.cart.name, "cart_price")
                }

                /**
                 * Will throw an exception for the developer if currency provided is null or not
                 * provided at all.
                 */
                currencyValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.cart.name,
                        InternalCurrencyCode::class.java.simpleName
                    )
                }

                else -> {

                    ECommerceConstants.isItemValueObjectValid(itemValue, EComEventType.cart)

                    if (currencyValue != itemValue.currency) {
                        ExceptionManager.throwCurrencyNotSameException(
                            EComEventType.cart.name,
                            "cart"
                        )
                    }

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    cartObject = CartObject(
                        cartIdValue!!, cartActionValue!!,
                        itemValue, cartPriceValue!!, currencyValue!!, meta
                    )

                    CFSetup().track(
                        ECommerceConstants.contentBlockName, EComEventType.cart.name,
                        cartObject, isUpdateImmediately
                    )
                }
            }
        }

    }
}
