package ai.causalfoundry.android.sdk.e_commerce.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.utils.Converters
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.DeliveryObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.CoordinatesObject
import ai.causalfoundry.android.sdk.e_commerce.event_types.DeliveryAction
import ai.causalfoundry.android.sdk.e_commerce.event_types.EComEventType
import ai.causalfoundry.android.sdk.e_commerce.utils.ECommerceConstants
import com.google.gson.Gson

/**
 * Created by Moiz Hassan on 22 March, 2023
 */

class CfLogDeliveryEvent {

    /**
     * CfLogDeliveryEvent is used to log the status for the delivery. It can be used to log the
     * delivered status of the order or a partial order. Details about the items in the specific
     * delivery should be provided in the catalog.
     */
    data class Builder(
        internal var orderIdValue: String? = null,
        internal var deliveryIdValue: String? = null,
        internal var actionValue: String? = null,
        internal var estDeliveryTsValue: Long? = null,
        internal var isUrgentValue: Boolean? = null,
        internal var deliveryCoordinatesObject: CoordinatesObject? = null,
        internal var dispatchCoordinatesObject: CoordinatesObject? = null,
        private var meta: Any? = null,
        private var isUpdateImmediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * setOrderId is required to associate the delivery event with the order. OrderId should
         * be a valid orderId and can be tracked from the checkout.
         */
        fun setOrderId(orderId: String) = apply { this.orderIdValue = orderId }

        /**
         * setDeliveryId is required to associate the rating obtained for the order. deliveryId should
         * be a valid deliveryId and can be tracked from the catalog for the items in that
         * specific delivery.
         */
        fun setDeliveryId(deliveryId: String) = apply { this.deliveryIdValue = deliveryId }

        /**
         * setDeliveryAction is required to set the delivery action for the log. For the order
         * being prepared for delivery or left the shipment center or delivered to the customer.
         */
        fun setDeliveryAction(action: DeliveryAction) = apply { this.actionValue = action.name }
        fun setDeliveryAction(action: String) = apply {
            if (CoreConstants.enumContains<DeliveryAction>(action)) {
                this.actionValue = action
            } else {
                ExceptionManager.throwEnumException(
                    EComEventType.delivery.name,
                    DeliveryAction::class.java.simpleName
                )
            }
        }

        /**
         * isUrgent is to mark the log if the delivery is scheduled to be an immediate or emergency
         * delivery.
         */
        fun isUrgent(isUrgent: Boolean) = apply { this.isUrgentValue = isUrgent }

        /**
         * The timestamp in time in milliseconds format for the date and time for which the
         * delivery is set to be scheduled
         */
        fun setDeliveryDateTime(deliveryTs: Long) = apply { this.estDeliveryTsValue = deliveryTs }
        fun setDeliveryDateTime(deliveryTsString: String) = apply {
            try {
                this.estDeliveryTsValue = deliveryTsString.toLong()
            } catch (ex: java.lang.Exception) {
                ExceptionManager.throwRuntimeException(
                    EComEventType.delivery.name,
                    "Unable to convert $deliveryTsString to Long"
                )
            }
        }

        /**
         * setDeliveryCoordinates can be used to pass the whole Coordinates Object or as a
         * Json String as well. You can use the POJO CoordinatesObject to parse the data
         * in the required format and pass that to this function as a string to log the event.
         * You can use Gson to convert the object to string but SDK will parse the Json string
         * back to POJO so pass it in the log. This method should be used with caution and is
         * suitable for react native bridge.
         */
        fun setDeliveryCoordinates(deliveryCoordinates: CoordinatesObject) = apply {
            this.deliveryCoordinatesObject = deliveryCoordinates
        }

        fun setDeliveryCoordinates(deliveryCoordinatesJsonString: String?) = apply {
            if (!deliveryCoordinatesJsonString.isNullOrEmpty()) {
                val deliveryCoordinatesObj: CoordinatesObject =
                    Gson().fromJson(deliveryCoordinatesJsonString, CoordinatesObject::class.java)
                this.deliveryCoordinatesObject = deliveryCoordinatesObj
            }
        }

        /**
         * setDispatchCoordinates can be used to pass the whole Coordinates Object or as a
         * Json String as well. You can use the POJO CoordinatesObject to parse the data
         * in the required format and pass that to this function as a string to log the event.
         * You can use Gson to convert the object to string but SDK will parse the Json string
         * back to POJO so pass it in the log. This method should be used with caution and is
         * suitable for react native bridge.
         */
        fun setDispatchCoordinates(dispatchCoordinates: CoordinatesObject) = apply {
            this.dispatchCoordinatesObject = dispatchCoordinates
        }

        fun setDispatchCoordinates(dispatchCoordinatesJsonString: String?) = apply {
            if (!dispatchCoordinatesJsonString.isNullOrEmpty()) {
                val dispatchCoordinatesObj: CoordinatesObject =
                    Gson().fromJson(dispatchCoordinatesJsonString, CoordinatesObject::class.java)
                this.dispatchCoordinatesObject = dispatchCoordinatesObj
            }
        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(updateImmediately: Boolean) =
            apply { this.isUpdateImmediately = updateImmediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw and exception if the orderId provided is null or no value is
                 * provided at all.
                 */
                orderIdValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.delivery.name,
                        "order_id"
                    )
                }

                /**
                 * Will throw and exception if the delivery_id provided is null or no value is
                 * provided at all.
                 */
                deliveryIdValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.delivery.name,
                        "delivery_id"
                    )
                }
                /**
                 * Will throw and exception if the delivery_action provided is null or no value is
                 * provided at all.
                 */
                actionValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.delivery.name,
                        DeliveryAction::class.java.simpleName
                    )
                }
                /**
                 * Will throw and exception if the is_urgent provided is null or no value is
                 * provided at all.
                 */
                isUrgentValue == null -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.delivery.name, "is_urgent"
                    )
                }

                /**
                 * Will throw and exception if the delivery_ts provided is null or no value is
                 * provided at all.
                 */
                estDeliveryTsValue == null || estDeliveryTsValue == 0L -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.delivery.name, "est_delivery_ts"
                    )
                }

                (estDeliveryTsValue!! < System.currentTimeMillis() - 10000L) -> {
                    ExceptionManager.throwIllegalStateException(
                        EComEventType.delivery.name, "Scheduled time should be in future"
                    )
                }

                (deliveryCoordinatesObject != null && (deliveryCoordinatesObject?.lat == 0f ||
                        deliveryCoordinatesObject?.lon == 0f)) -> {
                    ExceptionManager.throwIllegalStateException(
                        EComEventType.delivery.name, "Invalid Delivery Coordinates provided"
                    )
                }

                (dispatchCoordinatesObject != null && (dispatchCoordinatesObject?.lat == 0f ||
                        dispatchCoordinatesObject?.lon == 0f)) -> {
                    ExceptionManager.throwIllegalStateException(
                        EComEventType.delivery.name, "Invalid Dispatch Coordinates provided"
                    )
                }

                else -> {

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val deliveryObject = DeliveryObject(deliveryIdValue!!, orderIdValue!!,
                        actionValue!!, isUrgentValue!!,
                        Converters.convertMillisToTimeString(estDeliveryTsValue!!),
                        deliveryCoordinatesObject, dispatchCoordinatesObject, meta)
                    CFSetup().track(
                        ECommerceConstants.contentBlockName, EComEventType.delivery.name,
                        deliveryObject, isUpdateImmediately
                    )
                }
            }
        }
    }
}
