package ai.causalfoundry.android.sdk.e_commerce.utils

import ai.causalfoundry.android.sdk.core.catalog.catalog_models.CatalogSubject
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.BloodCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.DrugCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.FacilityCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.GroceryCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.MedicalEquipmentCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.OxygenCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.internal_catalog_models.InternalBloodCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.internal_catalog_models.InternalDrugModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.internal_catalog_models.InternalFacilityCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.internal_catalog_models.InternalGroceryCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.internal_catalog_models.InternalMedicalEquipmentCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.internal_catalog_models.InternalOxygenCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.ItemModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.ItemTypeModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.meta_models.BloodMetaModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.meta_models.OxygenMetaModel
import ai.causalfoundry.android.sdk.e_commerce.event_types.EComEventType
import ai.causalfoundry.android.sdk.e_commerce.event_types.InternalCurrencyCode
import ai.causalfoundry.android.sdk.e_commerce.event_types.ItemType
import ai.causalfoundry.android.sdk.e_commerce.event_types.SubscriptionStatus
import ai.causalfoundry.android.sdk.e_commerce.event_types.SubscriptionType

/**
 * Created by Moiz Hassan on 22 March, 2023
 */


object ECommerceConstants {

    // SDK API endpoints
    const val contentBlockName: String = "e-commerce"

    fun isItemValueObjectValid(itemValue: ItemModel, eventType: EComEventType) {

        val eventName = eventType.name

        if (itemValue.id.isEmpty()) {
            ExceptionManager.throwIsRequiredException(eventName, "item_id")
        } else if (itemValue.quantity < 0) {
            ExceptionManager.throwItemQuantityException(eventName)
        } else if (itemValue.price == null) {
            ExceptionManager.throwIsRequiredException(eventName, "item_price")
        } else if (itemValue.price == -1f) {
            ExceptionManager.throwIsRequiredException(eventName, "item_price")
        } else if (itemValue.currency.isEmpty()) {
            ExceptionManager.throwIsRequiredException(eventName, "item_currency")
        } else if (!CoreConstants.enumContains<InternalCurrencyCode>(itemValue.currency)) {
            ExceptionManager.throwEnumException(eventName, "CurrencyCode")
        } else if (itemValue.type.isEmpty()) {
            ExceptionManager.throwIsRequiredException(eventName, "item_type")
        } else if (!CoreConstants.enumContains<ItemType>(itemValue.type)) {
            ExceptionManager.throwEnumException(eventName, "ItemType")
        }else if (itemValue.type == ItemType.subscription.name && itemValue.subscription == null) {
            ExceptionManager.throwIsRequiredException(eventName, "subscription object")
        } else if (itemValue.type == ItemType.subscription.name && !CoreConstants.enumContains<SubscriptionStatus>(itemValue.subscription?.status!!)) {
            ExceptionManager.throwEnumException(eventName, "subscription status")
        } else if (itemValue.type == ItemType.subscription.name && !CoreConstants.enumContains<SubscriptionType>(itemValue.subscription?.type!!)) {
            ExceptionManager.throwEnumException(eventName, "subscription type")
        } else if (itemValue.type == ItemType.subscription.name && itemValue.subscription?.subcriptionItems.isNullOrEmpty()) {
            ExceptionManager.throwIsRequiredException(eventName, "subscription items")
        } else if (eventType == EComEventType.checkout && itemValue.type == ItemType.blood.name) {
            if (itemValue.meta == null) {
                ExceptionManager.throwIsRequiredException(eventName, "Blood Meta Properties")
            } else if (itemValue.meta !is BloodMetaModel) {
                ExceptionManager.throwEnumException(eventName, "Blood Meta Properties")
            }
        } else if (eventType == EComEventType.checkout && itemValue.type == ItemType.oxygen.name) {
            if (itemValue.meta == null) {
                ExceptionManager.throwIsRequiredException(eventName, "Oxygen Meta Properties")
            } else if (itemValue.meta !is OxygenMetaModel) {
                ExceptionManager.throwEnumException(eventName, "Oxygen Meta Properties")
            }
        }

        if (itemValue.type == ItemType.subscription.name && itemValue.subscription?.subcriptionItems!!.isNotEmpty()) {
            for (itemTypeObject in itemValue.subscription?.subcriptionItems!!){
                isItemTypeObjectValid(itemTypeObject, eventType)
            }
        }

    }

    fun isItemTypeObjectValid(itemValue: ItemTypeModel, eventType: EComEventType) {

        val eventName = eventType.name

        if (itemValue.itemId.isNullOrEmpty()) {
            ExceptionManager.throwIsRequiredException(eventName, "item_id")
        } else if (itemValue.itemType.isNullOrEmpty()) {
            ExceptionManager.throwIsRequiredException(eventName, "item_type")
        } else if (!CoreConstants.enumContains<ItemType>(itemValue.itemType)) {
            ExceptionManager.throwEnumException(eventName, "ItemType")
        }
    }

    fun verifyCatalogForDrug(
        drugId: String,
        drugCatalogModel: DrugCatalogModel
    ): InternalDrugModel {

        val catalogName = CatalogSubject.drug.name + " catalog"

        when {
            drugId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "drug Id")
            }

            drugCatalogModel.name.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "name")
            }
        }

        return InternalDrugModel(
            id = drugId,
            name = CoreConstants.checkIfNull(drugCatalogModel.name),
            marketId = CoreConstants.checkIfNull(drugCatalogModel.marketId),
            description = CoreConstants.checkIfNull(drugCatalogModel.description),
            supplierId = CoreConstants.checkIfNull(drugCatalogModel.supplierId),
            supplierName = CoreConstants.checkIfNull(drugCatalogModel.supplierName),
            producer = CoreConstants.checkIfNull(drugCatalogModel.producer),
            packaging = CoreConstants.checkIfNull(drugCatalogModel.packaging),
            activeIngredients = drugCatalogModel.activeIngredients ?: arrayListOf(),
            drugForm = CoreConstants.checkIfNull(drugCatalogModel.drugForm),
            drugStrength = CoreConstants.checkIfNull(drugCatalogModel.drugStrength),
            atcAnatomicalGroup = CoreConstants.checkIfNull(drugCatalogModel.atcAnatomicalGroup),
            otcOrEthical = CoreConstants.checkIfNull(drugCatalogModel.otcOrEthical)
        )

    }

    fun verifyCatalogForGrocery(
        itemId: String,
        groceryCatalogModel: GroceryCatalogModel
    ): InternalGroceryCatalogModel {

        val catalogName = CatalogSubject.grocery.name + " catalog"

        when {
            itemId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "grocery item Id")
            }

            groceryCatalogModel.name.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "name")
            }
        }

        return InternalGroceryCatalogModel(
            id = itemId,
            name = CoreConstants.checkIfNull(groceryCatalogModel.name),
            category = CoreConstants.checkIfNull(groceryCatalogModel.category),
            marketId = CoreConstants.checkIfNull(groceryCatalogModel.marketId),
            description = CoreConstants.checkIfNull(groceryCatalogModel.description),
            supplierId = CoreConstants.checkIfNull(groceryCatalogModel.supplierId),
            supplierName = CoreConstants.checkIfNull(groceryCatalogModel.supplierName),
            producer = CoreConstants.checkIfNull(groceryCatalogModel.producer),
            packaging = CoreConstants.checkIfNull(groceryCatalogModel.packaging),
            packagingSize = groceryCatalogModel.packagingSize ?: 0f,
            packagingUnits = CoreConstants.checkIfNull(groceryCatalogModel.packagingUnits),
            activeIngredients = groceryCatalogModel.activeIngredients ?: arrayListOf()
        )

    }

    fun verifyCatalogForBlood(
        itemId: String,
        bloodCatalogModel: BloodCatalogModel
    ): InternalBloodCatalogModel {

        val catalogName = CatalogSubject.blood.name + " catalog"

        when {
            itemId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "blood item Id")
            }

            bloodCatalogModel.bloodGroup.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "blood_group")
            }
        }

        return InternalBloodCatalogModel(
            id = itemId,
            marketId = CoreConstants.checkIfNull(bloodCatalogModel.marketId),
            bloodComponent = CoreConstants.checkIfNull(bloodCatalogModel.bloodComponent),
            bloodGroup = CoreConstants.checkIfNull(bloodCatalogModel.bloodGroup),
            packaging = CoreConstants.checkIfNull(bloodCatalogModel.packaging),
            packagingSize = bloodCatalogModel.packagingSize ?: 0f,
            packagingUnits = CoreConstants.checkIfNull(bloodCatalogModel.packagingUnits),
            supplierId = CoreConstants.checkIfNull(bloodCatalogModel.supplierId),
            supplierName = CoreConstants.checkIfNull(bloodCatalogModel.supplierName)
        )

    }

    fun verifyCatalogForOxygen(
        itemId: String,
        oxygenCatalogModel: OxygenCatalogModel
    ): InternalOxygenCatalogModel {

        val catalogName = CatalogSubject.oxygen.name + " catalog"

        when {
            itemId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "oxygen item Id")
            }
        }
        return InternalOxygenCatalogModel(
            id = itemId,
            marketId = CoreConstants.checkIfNull(oxygenCatalogModel.marketId),
            packaging = CoreConstants.checkIfNull(oxygenCatalogModel.packaging),
            packagingSize = oxygenCatalogModel.packagingSize ?: 0f,
            packagingUnits = CoreConstants.checkIfNull(oxygenCatalogModel.packagingUnits),
            supplierId = CoreConstants.checkIfNull(oxygenCatalogModel.supplierId),
            supplierName = CoreConstants.checkIfNull(oxygenCatalogModel.supplierName)
        )

    }

    fun verifyCatalogForMedicalEquipment(
        itemId: String,
        medicalEquipmentCatalogModel: MedicalEquipmentCatalogModel
    ): InternalMedicalEquipmentCatalogModel {

        val catalogName = CatalogSubject.medical_equipment.name + " catalog"

        when {
            itemId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "medical equipment item Id")
            }

            medicalEquipmentCatalogModel.name.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "name")
            }
        }
        return InternalMedicalEquipmentCatalogModel(
            id = itemId,
            name = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.name),
            description = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.description),
            marketId = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.marketId),
            supplierId = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.supplierId),
            supplierName = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.supplierName),
            producer = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.producer),
            packaging = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.packaging),
            packagingSize = medicalEquipmentCatalogModel.packagingSize ?: 0f,
            packagingUnits = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.packagingUnits),
            category = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.category)
        )

    }

    fun verifyCatalogForFacility(
        facilityId: String,
        facilityCatalogModel: FacilityCatalogModel
    ): InternalFacilityCatalogModel {

        val catalogName = CatalogSubject.facility.name + " catalog"

        when {
            facilityId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "facility Id")
            }

            facilityCatalogModel.name.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "name")
            }

            facilityCatalogModel.type.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "type")
            }
        }
        return InternalFacilityCatalogModel(
            id = facilityId,
            name = CoreConstants.checkIfNull(facilityCatalogModel.name),
            type = CoreConstants.checkIfNull(facilityCatalogModel.type),
            country = CoreConstants.checkIfNull(facilityCatalogModel.country),
            regionState = CoreConstants.checkIfNull(facilityCatalogModel.regionState),
            city = CoreConstants.checkIfNull(facilityCatalogModel.city),
            isActive = facilityCatalogModel.isActive ?: false,
            hasDelivery = facilityCatalogModel.hasDelivery ?: false,
            isSponsored = facilityCatalogModel.isSponsored ?: false
        )

    }

}
