package ai.causalfoundry.android.sdk.e_commerce.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.ItemVerificationObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.ItemInfoObject
import ai.causalfoundry.android.sdk.e_commerce.event_types.EComEventType
import ai.causalfoundry.android.sdk.e_commerce.event_types.ItemType
import ai.causalfoundry.android.sdk.e_commerce.event_types.ScanChannel
import ai.causalfoundry.android.sdk.e_commerce.event_types.ScanType
import ai.causalfoundry.android.sdk.e_commerce.utils.ECommerceConstants
import com.google.gson.Gson

/**
 * Created by Moiz Hassan on 04 April, 2023
 */

class CfLogItemVerificationEvent {

    /**
     * CfLogItemVerificationEvent is used to log the event an item is verified/scanned by the user.
     */

    data class Builder(
        internal var scanChannelValue: String? = null,
        internal var scanTypeValue: String? = null,
        internal var isSuccessfulValue: Boolean? = null,
        internal var itemInfoValue: ItemInfoObject? = null,
        private var meta: Any? = null,
        private var isUpdateImmediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * setScanChannel is for the providing scan channel used for verification.
         */
        fun setScanChannel(scanChannel: ScanChannel) = apply {
            this.scanChannelValue = scanChannel.toString()
        }

        fun setScanChannel(scanChannel: String) = apply {
            this.scanChannelValue = scanChannel
        }

        /**
         * setScanType is for the providing scan Type used for verification.
         */
        fun setScanType(scanType: ScanType) = apply {
            this.scanTypeValue = scanType.toString()
        }

        fun setScanType(scanType: String) = apply {
            this.scanTypeValue = scanType
        }

        /**
         * isSuccessful is for the providing if verification was successful or not.
         */
        fun isSuccessful(isSuccessful: Boolean) = apply {
            this.isSuccessfulValue = isSuccessful
        }

        /**
         * setItemInfo is for the providing item info details returned by the verification.
         * The object should be based on the ItemInfoObject or a string that can be
         * converted to the object with proper param names. in-case the names are not correct
         * the SDK will throw an exception. Below is the function for providing item as a string.
         */
        fun setItemInfo(itemInfo: ItemInfoObject) = apply {
            this.itemInfoValue = itemInfo
        }

        /**
         * setItemInfo is for the providing item info details returned by the verification.
         * The object should be based on the ItemInfoObject or a string that can be
         * converted to the object with proper param names. in-case the names are not correct
         * the SDK will throw an exception. Below is the function for providing item as a string.
         */
        fun setItemInfo(itemInfo: String) = apply {
            this.itemInfoValue = Gson().fromJson(itemInfo, ItemInfoObject::class.java)

        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(updateImmediately: Boolean) =
            apply { this.isUpdateImmediately = updateImmediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if scan_channel provided is null
                 * or not provided at all.
                 */
                scanChannelValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.ItemVerification.toString(),
                        "scan_channel"
                    )
                }

                !CoreConstants.enumContainsString<ScanChannel>(scanChannelValue!!) -> {
                    ExceptionManager.throwEnumException(
                        EComEventType.ItemVerification.toString(),
                        ScanChannel::class.java.simpleName
                    )
                }

                /**
                 * Will throw an exception for the developer if scan_type provided is null or not
                 * provided at all.
                 */
                scanTypeValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.ItemVerification.toString(),
                        "scan_type"
                    )
                }
                !CoreConstants.enumContainsString<ScanType>(scanTypeValue!!) -> {
                    ExceptionManager.throwEnumException(
                        EComEventType.ItemVerification.toString(),
                        ScanType::class.java.simpleName
                    )
                }

                /**
                 * Will throw an exception for the developer if is_successful provided is null or not
                 * provided at all.
                 */
                isSuccessfulValue == null -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.ItemVerification.toString(),
                        "is_successful"
                    )
                }

                isSuccessfulValue!! && itemInfoValue == null -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.ItemVerification.toString(),
                        "item_info"
                    )
                }

                isSuccessfulValue!! && !CoreConstants.enumContainsString<ItemType>(itemInfoValue!!.type) -> {
                    ExceptionManager.throwEnumException(
                        EComEventType.ItemVerification.toString(),
                        ItemType::class.java.simpleName
                    )
                }

                /**
                 * Will throw an exception for the developer if item id provided is null or not
                 * provided at all.
                 */

                isSuccessfulValue!! && itemInfoValue?.id.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.ItemVerification.toString(),
                        "item_info id"
                    )
                }

                /**
                 * Will throw an exception for the developer if item type provided is null or not
                 * provided at all.
                 */
                isSuccessfulValue!! && itemInfoValue?.type.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.ItemVerification.toString(),
                        "item_info type"
                    )
                }
                
                else -> {

                    if (!isSuccessfulValue!!){
                        itemInfoValue = null
                    }


                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val itemVerificationObject = ItemVerificationObject(
                        scanChannelValue!!, scanTypeValue!!, isSuccessfulValue!!, itemInfoValue, meta
                    )
                    CFSetup().track(
                        ECommerceConstants.contentBlockName, EComEventType.ItemVerification.toString(),
                        itemVerificationObject, isUpdateImmediately
                    )
                }
            }
        }

    }
}
