package ai.causalfoundry.android.sdk.e_commerce.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.ItemRequestObject
import ai.causalfoundry.android.sdk.e_commerce.event_types.EComEventType
import ai.causalfoundry.android.sdk.e_commerce.utils.ECommerceConstants

/**
 * Created by Moiz Hassan on 04 April, 2023
 */

class CfLogItemRequestEvent {

    /**
     * CfLogItemRequestEvent is used to log the event an item is requested.
     */

    data class Builder(
        internal var itemRequestIdValue: String? = null,
        internal var itemNameValue: String? = null,
        internal var manufacturerValue: String? = null,
        private var meta: Any? = null,
        private var isUpdateImmediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * setItemRequestId is for the providing Item name requested by the user.
         */
        fun setItemRequestId(itemRequestId: String) = apply {
            this.itemRequestIdValue = itemRequestId
        }

        /**
         * setItemName is for the providing Item name requested by the user.
         */
        fun setItemName(itemName: String) = apply {
            this.itemNameValue = itemName
        }

        /**
         * setItemManufacturer is for the providing manufacturer for the item requested by the user.
         */
        fun setItemManufacturer(manufacturer: String) = apply {
            this.manufacturerValue = manufacturer
        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(updateImmediately: Boolean) =
            apply { this.isUpdateImmediately = updateImmediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if id provided is null or not
                 * provided at all.
                 */
                itemRequestIdValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.ItemRequest.toString(),
                        "item_request_id"
                    )
                }

                /**
                 * Will throw an exception for the developer if name provided is null or not
                 * provided at all.
                 */
                itemNameValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.ItemRequest.toString(),
                        "item_name"
                    )
                }

                /**
                 * Will throw an exception for the developer if manufacturer provided is null or not
                 * provided at all.
                 */
                manufacturerValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.ItemRequest.toString(),
                        "manufacturer"
                    )
                }

                else -> {
                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val itemRequestObject = ItemRequestObject(
                        itemRequestIdValue!!, itemNameValue!!, manufacturerValue!!, meta
                    )
                    CFSetup().track(
                        ECommerceConstants.contentBlockName, EComEventType.ItemRequest.toString(),
                        itemRequestObject, isUpdateImmediately
                    )
                }
            }
        }

    }
}
