package ai.causalfoundry.android.sdk.e_commerce.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_commerce.catalog.CfEComCatalog
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.ViewItemObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.ItemModel
import ai.causalfoundry.android.sdk.e_commerce.event_types.EComEventType
import ai.causalfoundry.android.sdk.e_commerce.event_types.ItemAction
import ai.causalfoundry.android.sdk.e_commerce.utils.ECommerceConstants
import com.google.gson.Gson

/**
 * Created by Moiz Hassan on 22 March, 2023
 */

class CfLogItemEvent {

    /**
     * CfLogItemEvent is required to log item related events which included when an item is viewed
     * and when an item's detail is viewed.
     */

    data class Builder(
        internal var itemActionValue: String? = null,
        internal var itemValue: ItemModel = ItemModel("", 1, -1f, "", "", 0f, "", "", "", null, null),
        internal var catalogModelValue: Any? = null,
        private var meta: Any? = null,
        private var isUpdateImmediately: Boolean = CoreConstants.updateImmediately
    ) {
        private lateinit var itemObject: ViewItemObject

        /**
         * setItemViewAction is required to set Type for the type of log in this case, if a user is viewing
         * an item or it's details. setType log is used to define if the log is about the item
         * itself or if the user is going into the details. you can provide the values using 2
         * different methods, one is for enum based and the other is for string based.
         * Below is the method for the enum based approach.
         */
        fun setItemAction(itemAction: ItemAction) =
            apply { this.itemActionValue = itemAction.toString() }

        /**
         * setItemViewAction is required to set Type for the type of log in this case, if a user is viewing
         * an item or it's details. setType log is used to define if the log is about the item
         * itself or if the user id going into the details. you can provide the values using 2
         * different methods, one is for enum based and the other is for string based.
         * Below is the method for the string based approach. Remember to note that for string
         * input you need to use the same names as defined in the enum or else the events will
         * be discarded.
         */
        fun setItemAction(itemAction: String) = apply {
            this.itemActionValue = itemAction
        }

        /**
         * setItem can be used to pass the whole item as an object as well. You can use the POJO
         * ItemModel to parse the data int he required format and pass that to this function to
         * log the event.
         */
        fun setItem(item: ItemModel) = apply {
            this.itemValue = item
        }

        /**
         * setItem can be used to pass the whole item as a Json String object as well. You can use
         * the POJO ItemModel to parse the data int he required format and pass that to this
         * function as a string to log the event. You can use Gson to convert the object to string
         * but SDK will parse the Json string back to POJO so pass it in the log. This method
         * should be used with caution and is suitable for react native bridge.
         */
        fun setItem(itemJsonString: String) = apply {
            val item: ItemModel = Gson().fromJson(itemJsonString, ItemModel::class.java)
            this.itemValue = item
        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(updateImmediately: Boolean) =
            apply { this.isUpdateImmediately = updateImmediately }


        fun setCatalogProperties(catalogProperties: Any?) =
            apply {
                if (catalogProperties != null && catalogProperties != "") {
                    this.catalogModelValue =
                        ECommerceConstants.verifyCatalogObject(catalogProperties, itemValue)
                }
            }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw and exception if the type provided is null or no value is
                 * provided at all.
                 */
                itemActionValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.Item.toString(), ItemAction::class.java.simpleName
                    )
                }

                !CoreConstants.enumContainsString<ItemAction>(itemActionValue!!) -> {
                    ExceptionManager.throwEnumException(
                        EComEventType.Item.toString(),
                        ItemAction::class.java.simpleName
                    )
                }

                else -> {

                    if(ECommerceConstants.isItemValueObjectValid(itemValue, EComEventType.Item)){
                        /**
                         * Parsing the values into an object and passing to the setup block to queue
                         * the event based on its priority.
                         */

                        itemObject =
                            ViewItemObject(itemActionValue!!, itemValue, meta)

                        CFSetup().track(
                            ECommerceConstants.contentBlockName, EComEventType.Item.toString(),
                            itemObject, isUpdateImmediately
                        )
                    }

                    if (itemActionValue == ItemAction.View.toString() && this.catalogModelValue != null) {
                        CfEComCatalog.callCatalogAPI(
                            itemId = itemValue.id,
                            itemtype = itemValue.type,
                            catalogModel = this.catalogModelValue!!
                        )
                    }

                }
            }
        }
    }
}
