package ai.causalfoundry.android.sdk.e_commerce.impression_listerner

import ai.causalfoundry.android.sdk.core.impression_listener.CfCoreImpressionListener
import ai.causalfoundry.android.sdk.core.impression_listener.impression_model.RecyclerImpressionModel
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.core.utils.serializeToMap
import ai.causalfoundry.android.sdk.core.utils.toDataClass
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.BloodCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.DrugCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.FacilityCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.GroceryCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.MedicalEquipmentCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.OxygenCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.ViewItemObject
import ai.causalfoundry.android.sdk.e_commerce.event_types.EComEventType
import ai.causalfoundry.android.sdk.e_commerce.event_types.ItemAction
import ai.causalfoundry.android.sdk.e_commerce.event_types.ItemType
import ai.causalfoundry.android.sdk.e_commerce.impression_listerner.event_model.ItemImpressionModel
import ai.causalfoundry.android.sdk.e_commerce.utils.ECommerceConstants
import androidx.recyclerview.widget.RecyclerView
import com.google.gson.Gson

/**
 * Created by Moiz Hassan on 22 March, 2023
 */


object CfItemImpressionListener {

    private lateinit var recyclerViewValue: RecyclerView
    private lateinit var currentDataProviderValue: List<ItemImpressionModel>
    private var collectionViewId: String = ""

    @JvmStatic
    fun trackRecyclerView(
        recyclerView: RecyclerView,
        currentDataProvider: () -> List<ItemImpressionModel>,
    ) {

        recyclerViewValue = recyclerView
        currentDataProviderValue = currentDataProvider()

        callCoreImpressionListenerTrackRecyclerView()

    }

    // for native SDK  to track elements in a recyclerview
    private fun callCoreImpressionListenerTrackRecyclerView() {
        CfCoreImpressionListener.trackRecyclerView(recyclerViewValue, currentDataProvider = {
            currentDataProviderValue.map {
                RecyclerImpressionModel(
                    elementId = it.itemProperties.id,
                    contentBlock = ECommerceConstants.contentBlockName,
                    eventName = EComEventType.Item.toString(),
                    catalogSubject = it.itemProperties.type,
                    itemProperties = ViewItemObject(
                        ItemAction.Impression.toString(),
                        it.itemProperties, null
                    ),
                    catalogProperties = prepareCatalogObject(
                        it.itemProperties.id,
                        it.itemProperties.type,
                        it.catalogProperties
                    )
                )
            }
        }, null)
    }

    private fun prepareCatalogObject(
        itemId: String,
        itemType: String,
        catalogModel: Any?
    ): Any? {
        return when (itemType) {
            ItemType.Drug.toString() -> {
                ECommerceConstants.verifyCatalogForDrug(itemId, catalogModel as DrugCatalogModel)
            }

            ItemType.Grocery.toString() -> {
                ECommerceConstants.verifyCatalogForGrocery(
                    itemId,
                    catalogModel as GroceryCatalogModel
                )
            }

            ItemType.Facility.toString() -> {
                ECommerceConstants.verifyCatalogForFacility(
                    itemId,
                    catalogModel as FacilityCatalogModel
                )
            }

            ItemType.Blood.toString() -> {
                ECommerceConstants.verifyCatalogForBlood(
                    itemId,
                    catalogModel as BloodCatalogModel
                )
            }

            ItemType.Oxygen.toString() -> {
                ECommerceConstants.verifyCatalogForOxygen(
                    itemId,
                    catalogModel as OxygenCatalogModel
                )
            }

            ItemType.MedicalEquipment.toString() -> {
                ECommerceConstants.verifyCatalogForMedicalEquipment(
                    itemId,
                    catalogModel as MedicalEquipmentCatalogModel
                )
            }

            else -> {
                null
            }
        }

    }

    interface CurrentDataProvider {
        fun provideCurrentData(): List<ItemImpressionModel>
    }

    @JvmStatic
    fun startTrackingForJavaClass(
        recyclerView: RecyclerView,
        currentDataProvider: CurrentDataProvider,
    ) {
        trackRecyclerView(
            recyclerView,
            currentDataProvider::provideCurrentData,
        )
    }

    @JvmStatic
    fun onCollectionUpdated(
        collectionViewKey: String,
        currentDataProvider: List<ItemImpressionModel>
    ) {

        currentDataProviderValue = currentDataProvider
        collectionViewId = collectionViewKey

        callCoreCollectionUpdated()
    }

    @JvmStatic
    fun onCollectionUpdatedRN(
        collectionViewKey: String,
        currentDataProvider: String
    ) {

        val itemModels: Array<ItemImpressionModel> = Gson().fromJson(
            currentDataProvider,
            Array<ItemImpressionModel>::class.java
        )

        val itemList = java.util.ArrayList(itemModels.toMutableList())
        for (item in itemList) {
            try {
                when (item.itemProperties.type) {
                    ItemType.Drug.toString() -> {
                        item.catalogProperties = item.catalogProperties.serializeToMap()
                            .toDataClass() as DrugCatalogModel
                    }

                    ItemType.Grocery.toString() -> {
                        item.catalogProperties = item.catalogProperties.serializeToMap()
                            .toDataClass() as GroceryCatalogModel
                    }

                    ItemType.Facility.toString() -> {
                        item.catalogProperties = item.catalogProperties.serializeToMap()
                            .toDataClass() as FacilityCatalogModel
                    }

                    ItemType.Blood.toString() -> {
                        item.catalogProperties = item.catalogProperties.serializeToMap()
                            .toDataClass() as BloodCatalogModel
                    }

                    ItemType.Oxygen.toString() -> {
                        item.catalogProperties = item.catalogProperties.serializeToMap()
                            .toDataClass() as OxygenCatalogModel
                    }

                    ItemType.MedicalEquipment.toString() -> {
                        item.catalogProperties = item.catalogProperties.serializeToMap()
                            .toDataClass() as MedicalEquipmentCatalogModel
                    }
                }
            } catch (ex: java.lang.Exception) {
                ExceptionManager.throwIllegalStateException(
                    "impression_listener",
                    "Invalid catalog object provided"
                )
            }
            ECommerceConstants.isItemValueObjectValid(item.itemProperties, EComEventType.Item)
        }

        currentDataProviderValue = itemList
        collectionViewId = collectionViewKey

        if (itemList.isNotEmpty()) {
            callCoreCollectionUpdated()
        }
    }

    // for react native SDK  to track elements in a coming from RN wrapper
    private fun callCoreCollectionUpdated() {
        CfCoreImpressionListener.onCollectionUpdated(null, collectionViewId,
            currentDataProviderValue.map {
                RecyclerImpressionModel(
                    elementId = it.itemProperties.id,
                    contentBlock = ECommerceConstants.contentBlockName,
                    eventName = EComEventType.Item.toString(),
                    catalogSubject = it.itemProperties.type,
                    itemProperties = ViewItemObject(
                        ItemAction.Impression.toString(),
                        it.itemProperties, null
                    ),
                    catalogProperties = prepareCatalogObject(
                        it.itemProperties.id,
                        it.itemProperties.type,
                        it.catalogProperties
                    )
                )
            })
    }

}