package ai.causalfoundry.android.sdk.e_commerce.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.ItemReportObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.ItemTypeModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.ReportObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.StoreObject
import ai.causalfoundry.android.sdk.e_commerce.event_types.EComEventType
import ai.causalfoundry.android.sdk.e_commerce.utils.ECommerceConstants
import com.google.gson.Gson

/**
 * Created by Moiz Hassan on 03 April, 2023
 */

class CfLogItemReportEvent {

    /**
     * CfLogItemReportEvent is used to log the event an item is reported.
     */

    data class Builder(
        internal var itemObjectValue: ItemTypeModel? = null,
        internal var storeObjectValue: StoreObject? = null,
        internal var reportObjectValue: ReportObject? = null,
        private var meta: Any? = null,
        private var isUpdateImmediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * setItem is for the providing item Id and type for the item in question.
         * The object should be based on the ItemTypeModel or a string that can be
         * converted to the object with proper param names. in-case the names are not correct
         * the SDK will throw an exception. Below is the function for providing item as a string.
         */
        fun setItem(itemObject: ItemTypeModel) = apply {
            this.itemObjectValue = itemObject
        }

        /**
         * setItem is for the providing item Id and type for the item in question.
         * The object should be based on the ItemTypeModel or a string that can be
         * converted to the object with proper param names. in-case the names are not correct
         * the SDK will throw an exception. Below is the function for providing item as a string.
         */
        fun setItem(itemObject: String) = apply {
            setItem(Gson().fromJson(itemObject, ItemTypeModel::class.java))
        }

        /**
         * setStoreObject is for the providing store object details for the item report if any.
         * The object should be based on the StoreObject or a string that can be
         * converted to the object with proper param names. in-case the names are not correct
         * the SDK will throw an exception. Below is the function for providing item as a string.
         */
        fun setStoreObject(storeObject: StoreObject) = apply {
            this.storeObjectValue = storeObject
        }

        fun setStoreObject(storeObject: String) = apply {
            this.storeObjectValue = Gson().fromJson(storeObject, StoreObject::class.java)

        }

        /**
         * setReportObject is for the providing report object details for the item report if any.
         * The object should be based on the ReportObject or a string that can be
         * converted to the object with proper param names. In-case the names are not correct
         * the SDK will throw an exception. Below is the function for providing item as a string.
         */
        fun setReportObject(reportObject: ReportObject) = apply {
            this.reportObjectValue = reportObject
        }

        fun setReportObject(reportObject: String) = apply {
            this.reportObjectValue = Gson().fromJson(reportObject, ReportObject::class.java)
        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(updateImmediately: Boolean) =
            apply { this.isUpdateImmediately = updateImmediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw an exception for the developer if id provided is null or not
                 * provided at all.
                 */
                itemObjectValue == null -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.ItemReport.toString(),
                        "item_object"
                    )
                }

                /**
                 * Will throw an exception for the developer if store_object provided is null or not
                 * provided at all.
                 */
                storeObjectValue == null -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.ItemReport.toString(),
                        "store_object"
                    )
                }

                storeObjectValue!!.id.isEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.ItemReport.toString(),
                        "store_id"
                    )
                }

                /**
                 * Will throw an exception for the developer if report_object provided is null or not
                 * provided at all.
                 */
                reportObjectValue == null -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.ItemReport.toString(),
                        "report_object"
                    )
                }

                reportObjectValue!!.id.isEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.ItemReport.toString(),
                        "report_id"
                    )
                }

                reportObjectValue!!.shortDesc.isEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.ItemReport.toString(),
                        "report_short_desc"
                    )
                }

                else -> {

                    if (ECommerceConstants.isItemTypeObjectValid(
                            itemObjectValue!!,
                            EComEventType.ItemReport
                        )
                    ) {
                        /**
                         * Parsing the values into an object and passing to the setup block to queue
                         * the event based on its priority.
                         */
                        val itemReportObject = ItemReportObject(
                            itemObjectValue!!, storeObjectValue!!, reportObjectValue!!, meta
                        )
                        CFSetup().track(
                            ECommerceConstants.contentBlockName, EComEventType.ItemReport.toString(),
                            itemReportObject, isUpdateImmediately
                        )
                    }

                }
            }
        }

    }
}
