package ai.causalfoundry.android.sdk.e_commerce.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.event_types.CurrencyCode
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.CheckoutObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.ItemModel
import ai.causalfoundry.android.sdk.e_commerce.event_types.EComEventType
import ai.causalfoundry.android.sdk.e_commerce.event_types.ItemType
import ai.causalfoundry.android.sdk.e_commerce.event_types.ShopMode
import ai.causalfoundry.android.sdk.e_commerce.utils.ECommerceConstants
import android.util.Log
import com.google.gson.Gson
import kotlin.math.roundToInt

/**
 * Created by Moiz Hassan on 22 March, 2023
 */

class CfLogCheckoutEvent {

    /**
     * CfLogCheckoutEvent is required to log checkout event for orders. You can trigger this event
     * when the order has been placed or unable to place. you need to provide orderId in both cases.
     * If you don't have the orderId in case of not being successful then you can pass cartId for
     * that order.
     */

    data class Builder(
        internal var orderIdValue: String? = null,
        internal var cartIdValue: String? = null,
        internal var isSuccessfulValue: Boolean = true,
        internal var priceValue: Float? = null,
        internal var currencyValue: String? = null,
        internal var shopModeValue: String = ShopMode.Delivery.toString(),
        internal var itemListValue: ArrayList<ItemModel> = arrayListOf(),
        private var meta: Any? = null,
        private var isUpdateImmediately: Boolean = CoreConstants.updateImmediately
    ) {

        private lateinit var checkoutObject: CheckoutObject

        /**
         * setOrderId is required to log the orderId for the order being successful or failed
         * to place. In case of not having any orderId for failed cases you can pass the cartId
         * or a corresponding value that can be used to track in the catalog about the order
         * placement details.
         */
        fun setOrderId(orderId: String) = apply { this.orderIdValue = orderId }

        /**
         * setCartId can be used to log the cartId the event is logged for. It is recommended to
         * include the unique cartId for the cart items so that they can be tracked.
         */
        fun setCartId(cartId: String) = apply { this.cartIdValue = cartId }

        /**
         * isSuccessful is required to log the successful placement of the order. False in case of
         * order placement is not successful.
         */
        fun isSuccessful(isSuccessful: Boolean) = apply { this.isSuccessfulValue = isSuccessful }

        /**
         * setPrice is required to log the total price of the order being logged. Price format
         * should be in accordance to the currency selected.
         */
        fun setPrice(price: Float) = apply {
            this.priceValue = ((price * 100.0).roundToInt() / 100.0).toFloat()
        }
        fun setPrice(price: Int?) = apply { this.priceValue = price?.toFloat() }
        fun setPrice(price: Double) = apply {
            this.priceValue = ((price * 100.0).roundToInt() / 100.0).toFloat()
        }

        /**
         * setCurrency is required to log the currency for for the order logged. Currency should
         * be in ISO 4217 format. For Ease, SDK provides the enums to log the currency so that it
         * would be easy to log. You can also use the string function to provide the currency.
         * Below is the function for the logging currency using String. Remember to use the same
         * strings as provided in the enums or else the event will be discarded.
         */
        fun setCurrency(currency: String) = apply {
            this.currencyValue = currency
        }

        /**
         * setShopMode is required to log the mode which is used to by the user to shop the
         * product, either by delivery or pickup. By default the SDK offers the shop mode
         * to be delivery. You can pas the enum as well as String as well but that needs to
         * correspond the enums available.
         */
        fun setShopMode(shopMode: ShopMode) = apply { this.shopModeValue = shopMode.toString() }

        /**
         * setShopMode is required to log the mode which is used to by the user to shop the
         * product, either by delivery or pickup. By default the SDK offers the shop mode
         * to be delivery. You can pas the enum as well as String as well but that needs to
         * correspond the enums available.
         */
        fun setShopMode(shopMode: String) = apply {
            this.shopModeValue = shopMode
        }

        /**
         * addItem can be used to add the item being orders in to the checkout list. This log can
         * be used to add one item to the log at a time. Order item should be in a valid format.
         * With elements of the orderObject as:
         * ItemModel(itemID, type, price, currency, stock_status, quantity, promoId) Promo Id can be an
         * empty string or no value at all if the item does not have a promo offer that is obtained
         * by the user. You can add multiple addOrder functions to one checkout event to include
         * all the items in the order.
         */
        fun addItem(itemModel: ItemModel) = apply {
            this.itemListValue.add(itemModel)
        }

        /**
         * addItemList can be used to add the whole list to the log at once, the format should be
         * ArrayList<ItemModel> to log the event successfully. Order item should be in a valid format.
         * With elements of the orderObject as:
         * ItemModel(itemID, type, price, currency, stock_status, quantity, promoId) Promo Id can be an
         * empty string or no value at all if the item does not have a promo offer that is obtained
         * by the user. You should use only one addItemList with checkout event or else the list
         * will only save the later one.
         */
        fun addItemList(itemList: ArrayList<ItemModel>) = apply {
            this.itemListValue.addAll(itemList)
        }

        /**
         * addItemList can be used to add the whole list to the log at once, the format should be
         * ArrayList<ItemModel> to log the event successfully. But the input param is of type
         * string , this is special use case for react native logs where list can be passed as
         * Json string and can be used to log the event. Order item should be in a valid format.
         * With elements of the orderObject as:
         * ItemModel(itemID, type, price, currency, stock_status, quantity, promoId) Promo Id can be an
         * empty string or no value at all if the item does not have a promo offer that is obtained
         * by the user. You should use only one addItemList with checkout event or else the list
         * will only save the later one.
         */
        fun addItemList(itemList: String) = apply {
            this.itemListValue.clear()
            val itemModels: Array<ItemModel> = Gson().fromJson(
                itemList,
                Array<ItemModel>::class.java
            )
            for(item in itemModels){
                if(item.type == ItemType.Blood.toString() || item.type == ItemType.Oxygen.toString()){
                    val itemMeta = ECommerceConstants.parseMeta(item)
                    item.meta = itemMeta
                }
                this.itemListValue.add(item)
            }

        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(updateImmediately: Boolean) =
            apply { this.isUpdateImmediately = updateImmediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw and exception if the order_id provided is null or no value is
                 * provided at all.
                 */
                orderIdValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.Checkout.toString(),
                        "order_id"
                    )
                }

                /**
                 * Will throw and exception if the cart_id provided is null or no value is
                 * provided at all.
                 */
                cartIdValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.Checkout.toString(),
                        "cart_id"
                    )
                }

                /**
                 * Will throw and exception if the price provided is null or no value is
                 * provided at all.
                 */
                priceValue == null -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.Checkout.toString(),
                        "price_value"
                    )
                }

                /**
                 * Will throw and exception if the currency provided is null or no value is
                 * provided at all.
                 */
                currencyValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.Checkout.toString(),
                        "CurrencyCode"
                    )
                }

                !CoreConstants.enumContains<CurrencyCode>(currencyValue!!) -> {
                    ExceptionManager.throwEnumException(EComEventType.Checkout.toString(), "CurrencyCode")
                }

                /**
                 * Will throw and exception if the shop mode provided is null or no value is
                 * provided at all.
                 */
                shopModeValue.isEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.Checkout.toString(),
                        CurrencyCode::class.java.simpleName
                    )
                }

                !CoreConstants.enumContainsString<ShopMode>(shopModeValue) -> {
                    ExceptionManager.throwEnumException(EComEventType.Checkout.toString(), "ShopMode")
                }

                /**
                 * Will throw and exception if the item_list provided is null or no value is
                 * provided at all for the order Items.
                 */
                itemListValue.isEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        EComEventType.Checkout.toString(),
                        "item_list"
                    )
                }

                else -> {

                    var allItemsValid = true

                    for (item in itemListValue) {

                        if (!ECommerceConstants.isItemValueObjectValid(item, EComEventType.Checkout)) {
                            allItemsValid = false
                            break
                        } else if (currencyValue != item.currency) {
                            ExceptionManager.throwCurrencyNotSameException(
                                EComEventType.Checkout.toString(),
                                "checkout"
                            )
                            allItemsValid = false
                            break
                        }
                    }

                    if(allItemsValid){
                        /**
                         * Parsing the values into an object and passing to the setup block to queue
                         * the event based on its priority.
                         */
                        checkoutObject = CheckoutObject(
                            orderIdValue!!, cartIdValue!!, isSuccessfulValue, priceValue,
                            currencyValue, shopModeValue, itemListValue.distinctBy { it.id }, meta
                        )
                        CFSetup().track(
                            ECommerceConstants.contentBlockName, EComEventType.Checkout.toString(),
                            checkoutObject, isUpdateImmediately
                        )
                    }
                }
            }
        }

    }
}
