package ai.causalfoundry.android.sdk.e_commerce.utils

import ai.causalfoundry.android.sdk.core.catalog.catalog_models.CatalogSubject
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.BloodCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.DrugCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.FacilityCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.GroceryCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.MedicalEquipmentCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.catalog.catalog_models.OxygenCatalogModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.ItemModel
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.ItemTypeModel
import ai.causalfoundry.android.sdk.e_commerce.event_types.EComEventType
import ai.causalfoundry.android.sdk.e_commerce.event_types.ItemType
import ai.causalfoundry.android.sdk.e_commerce.event_types.SubscriptionStatus
import ai.causalfoundry.android.sdk.e_commerce.event_types.SubscriptionType
import com.google.gson.Gson

/**
 * Created by Moiz Hassan on 22 March, 2023
 */


internal object ECommerceConstants {

    // SDK API endpoints
    const val contentBlockName: String = "e-commerce"

    fun isItemValueObjectValid(itemValue: ItemModel, eventType: EComEventType): Boolean {

        val eventName = eventType.toString()

        if (itemValue.id.isEmpty()) {
            return ExceptionManager.throwIsRequiredException(eventName, "item_id")
        } else if (itemValue.quantity < 0) {
            return ExceptionManager.throwItemQuantityException(eventName)
        } else if (itemValue.price < 0) {
            return ExceptionManager.throwIsRequiredException(eventName, "item_price")
        } else if (itemValue.type == ItemType.Subscription && itemValue.subscription == null) {
            return ExceptionManager.throwIsRequiredException(eventName, "subscription object")
        } else if (itemValue.type == ItemType.Subscription && !CoreConstants.enumContainsString<SubscriptionStatus>(
                itemValue.subscription?.status!!
            )
        ) {
            return ExceptionManager.throwEnumException(eventName, "subscription status")
        } else if (itemValue.type == ItemType.Subscription && !CoreConstants.enumContainsString<SubscriptionType>(
                itemValue.subscription?.type!!
            )
        ) {
            return ExceptionManager.throwEnumException(eventName, "subscription type")
        } else if (itemValue.type == ItemType.Subscription && itemValue.subscription?.subscriptionItems.isNullOrEmpty()) {
            return ExceptionManager.throwIsRequiredException(eventName, "subscription items")
        }
//        else if (eventType == EComEventType.Checkout && itemValue.type == ItemType.Blood.toString()) {
//            if (itemValue.meta == null) {
//                return ExceptionManager.throwIsRequiredException(eventName, "Blood Meta Properties")
//            } else if (itemValue.meta !is BloodMetaModel) {
//                return ExceptionManager.throwEnumException(eventName, "Blood Meta Properties")
//            }
//        } else if (eventType == EComEventType.Checkout && itemValue.type == ItemType.Oxygen.toString()) {
//            if (itemValue.meta == null) {
//                return ExceptionManager.throwIsRequiredException(eventName, "Oxygen Meta Properties")
//            } else if (itemValue.meta !is OxygenMetaModel) {
//                return ExceptionManager.throwEnumException(eventName, "Oxygen Meta Properties")
//            }
//        }
        else if (itemValue.type == ItemType.Subscription && itemValue.subscription?.subscriptionItems!!.isNotEmpty()) {
            for (itemTypeObject in itemValue.subscription?.subscriptionItems!!) {
                return isItemTypeObjectValid(itemTypeObject, eventType)
            }
        }

        return true

    }

    fun isItemTypeObjectValid(itemValue: ItemTypeModel, eventType: EComEventType): Boolean {

        val eventName = eventType.toString()

        if (itemValue.itemId.isNullOrEmpty()) {
            return ExceptionManager.throwIsRequiredException(eventName, "item_id")
        }
        return true
    }


    fun verifyCatalogObject(catalogProperties: Any, itemValue: ItemModel): Any {
        try {
            if (catalogProperties is String) {
                when (itemValue.type) {
                    ItemType.Drug -> {
                        return Gson().fromJson(
                            catalogProperties,
                            DrugCatalogModel::class.java
                        )
                    }

                    ItemType.Blood -> {
                        return Gson().fromJson(
                            catalogProperties,
                            BloodCatalogModel::class.java
                        )
                    }

                    ItemType.Oxygen -> {
                        return Gson().fromJson(
                            catalogProperties,
                            OxygenCatalogModel::class.java
                        )
                    }

                    ItemType.MedicalEquipment -> {
                        return Gson().fromJson(
                            catalogProperties,
                            MedicalEquipmentCatalogModel::class.java
                        )
                    }

                    else -> {
                        return ExceptionManager.throwIllegalStateException(
                            EComEventType.Item.toString(),
                            "Please use correct catalog properties with provided item type"
                        )
                    }
                }
            } else {
                return catalogProperties
            }
        } catch (ex: Exception) {
            return ExceptionManager.throwIllegalStateException(
                EComEventType.Item.toString(),
                "Please use correct catalog properties with provided item type"
            )
        }
    }


    fun verifyCatalogForDrug(
        drugCatalogModel: DrugCatalogModel
    ): DrugCatalogModel {

        val catalogName = CatalogSubject.Drug.toString() + " catalog"

        when {
            drugCatalogModel.drugId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "drug Id")
            }

            drugCatalogModel.name.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "drug name")
            }
        }

        return DrugCatalogModel(
            drugId = drugCatalogModel.drugId,
            name = drugCatalogModel.name,
            marketId = CoreConstants.checkIfNull(drugCatalogModel.marketId),
            description = CoreConstants.checkIfNull(drugCatalogModel.description),
            supplierId = CoreConstants.checkIfNull(drugCatalogModel.supplierId),
            supplierName = CoreConstants.checkIfNull(drugCatalogModel.supplierName),
            producer = CoreConstants.checkIfNull(drugCatalogModel.producer),
            packaging = CoreConstants.checkIfNull(drugCatalogModel.packaging),
            activeIngredients = drugCatalogModel.activeIngredients ?: arrayListOf(),
            drugForm = CoreConstants.checkIfNull(drugCatalogModel.drugForm),
            drugStrength = CoreConstants.checkIfNull(drugCatalogModel.drugStrength),
            atcAnatomicalGroup = CoreConstants.checkIfNull(drugCatalogModel.atcAnatomicalGroup),
            otcOrEthical = CoreConstants.checkIfNull(drugCatalogModel.otcOrEthical)
        )

    }

    fun verifyCatalogForGrocery(
        groceryCatalogModel: GroceryCatalogModel
    ): GroceryCatalogModel {

        val catalogName = CatalogSubject.Grocery.toString() + " catalog"

        when {
            groceryCatalogModel.groceryId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "grocery item Id")
            }

            groceryCatalogModel.name.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "grocery name")
            }
        }

        return GroceryCatalogModel(
            groceryId = groceryCatalogModel.groceryId,
            name = groceryCatalogModel.name,
            category = CoreConstants.checkIfNull(groceryCatalogModel.category),
            marketId = CoreConstants.checkIfNull(groceryCatalogModel.marketId),
            description = CoreConstants.checkIfNull(groceryCatalogModel.description),
            supplierId = CoreConstants.checkIfNull(groceryCatalogModel.supplierId),
            supplierName = CoreConstants.checkIfNull(groceryCatalogModel.supplierName),
            producer = CoreConstants.checkIfNull(groceryCatalogModel.producer),
            packaging = CoreConstants.checkIfNull(groceryCatalogModel.packaging),
            packagingSize = groceryCatalogModel.packagingSize ?: 0f,
            packagingUnits = CoreConstants.checkIfNull(groceryCatalogModel.packagingUnits),
            activeIngredients = groceryCatalogModel.activeIngredients ?: arrayListOf()
        )

    }

    fun verifyCatalogForBlood(
        bloodCatalogModel: BloodCatalogModel
    ): BloodCatalogModel {

        val catalogName = CatalogSubject.Blood.toString() + " catalog"

        when {
            bloodCatalogModel.itemId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "blood item Id")
            }

            bloodCatalogModel.bloodGroup.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "blood_group")
            }
        }

        return BloodCatalogModel(
            itemId = bloodCatalogModel.itemId,
            marketId = CoreConstants.checkIfNull(bloodCatalogModel.marketId),
            bloodComponent = CoreConstants.checkIfNull(bloodCatalogModel.bloodComponent),
            bloodGroup = bloodCatalogModel.bloodGroup,
            packaging = CoreConstants.checkIfNull(bloodCatalogModel.packaging),
            packagingSize = bloodCatalogModel.packagingSize ?: 0f,
            packagingUnits = CoreConstants.checkIfNull(bloodCatalogModel.packagingUnits),
            supplierId = CoreConstants.checkIfNull(bloodCatalogModel.supplierId),
            supplierName = CoreConstants.checkIfNull(bloodCatalogModel.supplierName)
        )

    }

    fun verifyCatalogForOxygen(
        oxygenCatalogModel: OxygenCatalogModel
    ): OxygenCatalogModel {

        val catalogName = CatalogSubject.Oxygen.toString() + " catalog"

        when {
            oxygenCatalogModel.itemId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "oxygen item Id")
            }
        }
        return OxygenCatalogModel(
            itemId = oxygenCatalogModel.itemId,
            marketId = CoreConstants.checkIfNull(oxygenCatalogModel.marketId),
            packaging = CoreConstants.checkIfNull(oxygenCatalogModel.packaging),
            packagingSize = oxygenCatalogModel.packagingSize ?: 0f,
            packagingUnits = CoreConstants.checkIfNull(oxygenCatalogModel.packagingUnits),
            supplierId = CoreConstants.checkIfNull(oxygenCatalogModel.supplierId),
            supplierName = CoreConstants.checkIfNull(oxygenCatalogModel.supplierName)
        )

    }

    fun verifyCatalogForMedicalEquipment(
        medicalEquipmentCatalogModel: MedicalEquipmentCatalogModel
    ): MedicalEquipmentCatalogModel {

        val catalogName = CatalogSubject.MedicalEquipment.toString() + " catalog"

        when {
            medicalEquipmentCatalogModel.itemId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "medical equipment item Id")
            }

            medicalEquipmentCatalogModel.name.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "medical equipment name")
            }
        }
        return MedicalEquipmentCatalogModel(
            itemId = medicalEquipmentCatalogModel.itemId,
            name = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.name),
            description = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.description),
            marketId = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.marketId),
            supplierId = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.supplierId),
            supplierName = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.supplierName),
            producer = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.producer),
            packaging = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.packaging),
            packagingSize = medicalEquipmentCatalogModel.packagingSize ?: 0f,
            packagingUnits = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.packagingUnits),
            category = CoreConstants.checkIfNull(medicalEquipmentCatalogModel.category)
        )

    }

    fun verifyCatalogForFacility(
        facilityCatalogModel: FacilityCatalogModel
    ): FacilityCatalogModel {

        val catalogName = CatalogSubject.Facility.toString() + " catalog"

        when {
            facilityCatalogModel.facilityId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "facility Id")
            }

            facilityCatalogModel.name.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "name")
            }

            facilityCatalogModel.type.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(catalogName, "type")
            }
        }
        return FacilityCatalogModel(
            facilityId = facilityCatalogModel.facilityId,
            name = facilityCatalogModel.name,
            type = facilityCatalogModel.type,
            country = CoreConstants.checkIfNull(facilityCatalogModel.country),
            regionState = CoreConstants.checkIfNull(facilityCatalogModel.regionState),
            city = CoreConstants.checkIfNull(facilityCatalogModel.city),
            isActive = facilityCatalogModel.isActive ?: false,
            hasDelivery = facilityCatalogModel.hasDelivery ?: false,
            isSponsored = facilityCatalogModel.isSponsored ?: false
        )

    }


}
