package ai.causalfoundry.android.sdk.e_commerce.validators

import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_commerce.event_models.EComEventMappers
import ai.causalfoundry.android.sdk.e_commerce.event_models.InternalCancelCheckoutObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.InternalCartObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.InternalCheckoutObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.InternalDeliveryObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.InternalItemReportObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.InternalItemVerificationObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.InternalViewItemObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.CancelCheckoutObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.CartObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.CheckoutObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.DeliveryObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.ItemReportObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.ItemRequestObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.ItemVerificationObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.event_objects.ViewItemObject
import ai.causalfoundry.android.sdk.e_commerce.event_models.item_objects.ItemModel
import ai.causalfoundry.android.sdk.e_commerce.event_types.EComEventType
import ai.causalfoundry.android.sdk.e_commerce.utils.ECommerceConstants

internal object EComEventValidator {

    fun validateItemObject(logObject: Any): InternalViewItemObject? {
        if (logObject !is ViewItemObject) {
            ExceptionManager.throwInvalidException(
                EComEventType.Item.toString(),
                ViewItemObject::class.java.simpleName
            )
            return null
        }

        return if (
            ECommerceConstants.isItemValueObjectValid(logObject.item, EComEventType.Item)
        ) {
            EComEventMappers.mapViewItemEventToInternalViewItemEvent(logObject)
        } else {
            null
        }
    }

    fun validateCartObject(logObject: Any): InternalCartObject? {
        if (logObject !is CartObject) {
            ExceptionManager.throwInvalidException(
                EComEventType.Cart.toString(),
                CartObject::class.java.simpleName
            )
            return null
        }
        when {
            logObject.cartId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(EComEventType.Cart.toString(), "cart_id")
            }

            logObject.cartPrice < 0 -> {
                ExceptionManager.throwIsRequiredException(
                    EComEventType.Cart.toString(),
                    "cart_price"
                )
            }

            (logObject.currency != logObject.item.currency) -> {
                ExceptionManager.throwCurrencyNotSameException(
                    EComEventType.Cart.toString(),
                    "cart"
                )
            }

            else -> {
                if (ECommerceConstants.isItemValueObjectValid(logObject.item, EComEventType.Cart)) {
                    return EComEventMappers.mapCartObjectToInternalCartObject(logObject)
                }
            }
        }
        return null
    }

    fun validateCheckoutObject(logObject: Any): InternalCheckoutObject? {
        if (logObject !is CheckoutObject) {
            ExceptionManager.throwInvalidException(
                EComEventType.Checkout.toString(),
                CheckoutObject::class.java.simpleName
            )
            return null
        }
        when {
            logObject.orderId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(
                    EComEventType.Checkout.toString(),
                    "order_id"
                )
            }

            logObject.cartId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(
                    EComEventType.Checkout.toString(),
                    "cart_id"
                )
            }

            logObject.cartPrice < 0 -> {
                ExceptionManager.throwIsRequiredException(
                    EComEventType.Checkout.toString(),
                    "cart_price"
                )
            }

            logObject.itemList.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(
                    EComEventType.Checkout.toString(),
                    "item_list"
                )
            }

            else -> {

                var allItemsValid = true
                for (item in logObject.itemList) {
                    if (!ECommerceConstants.isItemValueObjectValid(item, EComEventType.Checkout)) {
                        allItemsValid = false
                        break
                    } else if (logObject.currency != item.currency) {
                        ExceptionManager.throwCurrencyNotSameException(
                            EComEventType.Checkout.toString(),
                            "checkout"
                        )
                        allItemsValid = false
                        break
                    }
                }
                if (allItemsValid) {
                    return EComEventMappers.mapCheckoutObjectToInternalCheckoutObject(logObject)
                }
            }
        }
        return null
    }

    fun validateCancelCheckoutObject(logObject: Any): InternalCancelCheckoutObject? {
        if (logObject !is CancelCheckoutObject) {
            ExceptionManager.throwInvalidException(
                EComEventType.CancelCheckout.toString(),
                CancelCheckoutObject::class.java.simpleName
            )
            return null
        }
        when {
            logObject.checkoutId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(
                    EComEventType.CancelCheckout.toString(),
                    "checkout_id"
                )
            }

            logObject.itemsList.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(
                    EComEventType.CancelCheckout.toString(),
                    "items_list"
                )
            }

            else -> {
                val allItemsValid = logObject.itemsList.all { item ->
                    ECommerceConstants.isItemTypeObjectValid(item, EComEventType.CancelCheckout)
                }
                if (allItemsValid) {
                    return EComEventMappers.mapCancelCheckoutObjectToInternalCancelCheckoutObject(
                        logObject
                    )
                }
            }
        }
        return null
    }

    fun validateDeliveryObject(logObject: Any): InternalDeliveryObject? {
        if (logObject !is DeliveryObject) {
            ExceptionManager.throwInvalidException(
                EComEventType.Delivery.toString(),
                DeliveryObject::class.java.simpleName
            )
            return null
        }
        when {
            logObject.orderId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(
                    EComEventType.Delivery.toString(),
                    "order_id"
                )
            }

            logObject.deliveryId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(
                    EComEventType.Delivery.toString(),
                    "delivery_id"
                )
            }

            logObject.deliveryTs <= 0L -> {
                ExceptionManager.throwIsRequiredException(
                    EComEventType.Delivery.toString(), "est_delivery_ts"
                )
            }

            (logObject.deliveryCoordinates != null && (logObject.deliveryCoordinates?.lat == 0.0 ||
                    logObject.deliveryCoordinates?.lon == 0.0)) -> {
                ExceptionManager.throwIllegalStateException(
                    EComEventType.Delivery.toString(), "Invalid Delivery Coordinates provided"
                )
            }

            (logObject.dispatchCoordinates != null && (logObject.dispatchCoordinates?.lat == 0.0 ||
                    logObject.dispatchCoordinates?.lon == 0.0)) -> {
                ExceptionManager.throwIllegalStateException(
                    EComEventType.Delivery.toString(), "Invalid Dispatch Coordinates provided"
                )
            }

            else -> {
                return EComEventMappers.mapDeliveryObjectToInternalDeliveryObject(
                    logObject
                )
            }
        }
        return null
    }

    fun validateItemReportObject(logObject: Any): InternalItemReportObject? {
        if (logObject !is ItemReportObject) {
            ExceptionManager.throwInvalidException(
                EComEventType.ItemReport.toString(),
                ItemReportObject::class.java.simpleName
            )
            return null
        }
        when {

            logObject.storeInfo.id.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(
                    EComEventType.ItemReport.toString(),
                    "store_id"
                )
            }

            logObject.reportInfo.id.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(
                    EComEventType.ItemReport.toString(),
                    "report_id"
                )
            }

            logObject.reportInfo.shortDesc.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(
                    EComEventType.ItemReport.toString(),
                    "report_short_desc"
                )
            }

            else -> {

                if (ECommerceConstants.isItemTypeObjectValid(
                        logObject.item,
                        EComEventType.ItemReport
                    )
                ) {
                    return EComEventMappers.mapItemReportObjectToInternalItemReportObject(
                        logObject
                    )
                }

            }
        }
        return null
    }

    fun validateItemRequestObject(logObject: Any): ItemRequestObject? {
        if (logObject !is ItemRequestObject) {
            ExceptionManager.throwInvalidException(
                EComEventType.ItemRequest.toString(),
                ItemRequestObject::class.java.simpleName
            )
            return null
        }
        when {

            logObject.itemRequestId.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(
                    EComEventType.ItemRequest.toString(),
                    "item_request_id"
                )
            }

            logObject.itemName.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(
                    EComEventType.ItemRequest.toString(),
                    "item_name"
                )
            }

            logObject.manufacturer.isEmpty() -> {
                ExceptionManager.throwIsRequiredException(
                    EComEventType.ItemRequest.toString(),
                    "manufacturer"
                )
            }

            else -> {
                return logObject
            }
        }
        return null
    }

    fun validateItemVerificationObject(logObject: Any): InternalItemVerificationObject? {
        if (logObject !is ItemVerificationObject) {
            ExceptionManager.throwInvalidException(
                EComEventType.ItemVerification.toString(),
                ItemVerificationObject::class.java.simpleName
            )
            return null
        }
        when {

            logObject.isSuccessful && logObject.itemInfo == null -> {
                ExceptionManager.throwIsRequiredException(
                    EComEventType.ItemVerification.toString(),
                    "item_info"
                )
            }

            /**
             * Will throw an exception for the developer if item id provided is null or not
             * provided at all.
             */

            logObject.isSuccessful && logObject.itemInfo?.itemId.isNullOrEmpty() -> {
                ExceptionManager.throwIsRequiredException(
                    EComEventType.ItemVerification.toString(),
                    "item_info id"
                )
            }

            else -> {
                if (!logObject.isSuccessful){
                    logObject.itemInfo = null
                }
                return EComEventMappers.mapItemVerificationToInternalItemVerificationObject(logObject)
            }
        }
        return null
    }

}