package ai.causalfoundry.android.sdk.e_learning.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_learning.event_models.event_objects.ModuleObject
import ai.causalfoundry.android.sdk.e_learning.event_types.ELearnEventType
import ai.causalfoundry.android.sdk.e_learning.event_types.ExamAction
import ai.causalfoundry.android.sdk.e_learning.event_types.ModuleLogAction
import ai.causalfoundry.android.sdk.e_learning.utils.ELearningConstants

/**
 * Created by Moiz Hassan on 23 March, 2023
 */

class CfLogModuleEvent {

    /**
     * CfLogModuleEvent is required to log actions related to e-learning modules which includes
     * the log for user viewing the module, starting and finishing the module and also user
     * viewing the content page as well.
     */
    data class Builder(
        var module_id: String? = null,
        var progress: Int? = null,
        var action: String? = null,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * setModuleId is required to log moduleId for the module being opened by the user.
         * module Id should be in a string format and must be in accordance to the catalog
         * provided.
         */
        fun setModuleId(module_id: String) = apply { this.module_id = module_id }

        /**
         * setModuleProgress is required to pass current progress of the module which the user is
         * viewing. Progress value is the percentage complete for the module that needs to passed
         * as an integer.
         */
        fun setModuleProgress(progress: Int?) = apply { this.progress = progress }

        /**
         * setModuleAction is required to set the Action type for the module event. SDK provides
         * enum classes to support available log types. 1 main is view.
         * SDK provides 2 approaches to log this event, one being enum type and the other is
         * string type. Below is the function to log type using enum.
         */
        fun setModuleAction(action: ModuleLogAction) = apply { this.action = action.name }

        /**
         * setModuleAction is required to set the Action type for the module event. SDK provides
         * enum classes to support available log types. 1 main is view.
         * SDK provides 2 approaches to log this event, one being enum type and the other is
         * string type. Below is the function to log type using string. Remember to note that
         * values provided using string should be the same as provided in enum or else the
         * events will be discarded.
         */
        fun setModuleAction(action: String?) = apply {
            if (action != null) {
                if (CoreConstants.enumContains<ModuleLogAction>(action)) {
                    this.action = action
                } else {
                    ExceptionManager.throwEnumException(ModuleLogAction::class.java.simpleName)
                }
            } else {
                this.action = action
            }
        }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw and exception if the moduleId provided is null or no value is
                 * provided at all.
                 */
                module_id.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException("module_id")
                }

                /**
                 * Will throw and exception if the action provided is null or no value is
                 * provided at all.
                 */
                action.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(ExamAction::class.java.simpleName)
                }

                else -> {

                    /**
                     * Parsing the values into an object and passing to the setup block to queue
                     * the event based on its priority.
                     */
                    val moduleObject = ModuleObject(module_id!!, progress, action!!, meta)
                    CFSetup().track(
                        ELearningConstants.contentBlockName, ELearnEventType.module.name,
                        moduleObject, update_immediately
                    )
                }
            }
        }
    }
}
