package ai.causalfoundry.android.sdk.e_learning.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_learning.event_models.event_objects.QuestionObject
import ai.causalfoundry.android.sdk.e_learning.event_types.ELearnEventType
import ai.causalfoundry.android.sdk.e_learning.event_types.QuestionAction
import ai.causalfoundry.android.sdk.e_learning.utils.ELearningConstants

/**
 * Created by Moiz Hassan on 23 March, 2023
 */

class CfLogQuestionEvent {

    /**
     * CfLogQuestionEvent is required to log user answers to the questions. To log this event you
     * need to provide the question Id that User has attempted and also the id for the answer the
     * user has selected.
     */

    data class Builder(
        var question_id: String? = null,
        var exam_id: String? = null,
        var action: String? = null,
        var answer_id: String? = null,
        private var meta: Any? = null,
        private var update_immediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * setQuestionId is required to log questionId for the Question on which user has attempted.
         * Question Id should be in a string format and must be in accordance to the catalog
         * provided.
         */
        fun setQuestionId(question_id: String) = apply { this.question_id = question_id }

        /**
         * setExamId is required to log exam_id for the exam the Question belongs to.
         * exam_id should be in a string format and must be in accordance to the catalog
         * provided.
         */
        fun setExamId(exam_id: String) = apply { this.exam_id = exam_id }

        /**
         * setQuestionAction is required to set the Action type for the Question event. SDK provides
         * enum classes to support available log types. 2 main are answer and skip.
         * SDK provides 2 approaches to log this event, one being enum type and the other is
         * string type. Below is the function to log type using enum.
         */
        fun setQuestionAction(action: QuestionAction) = apply { this.action = action.name }

        /**
         * setQuestionAction is required to set the Action type for the Question event. SDK provides
         * enum classes to support available log types. 2 main are answer and skip.
         * SDK provides 2 approaches to log this event, one being enum type and the other is
         * string type. Below is the function to log type using string. Remember to note that
         * values provided using string should be the same as provided in enum or else the
         * events will be discarded.
         */
        fun setQuestionAction(action: String?) = apply {
            if (action != null) {
                if (CoreConstants.enumContains<QuestionAction>(action)) {
                    this.action = action
                } else {
                    ExceptionManager.throwEnumException(
                        ELearnEventType.question.name,
                        QuestionAction::class.java.simpleName
                    )
                }
            } else {
                this.action = action
            }
        }

        /**
         * setAnswerId is required to log answerId for the answer provided by the user for
         * the Question on which user has attempted. Answer Id should be in a string format and
         * must be in accordance to the catalog provided.
         */
        fun setAnswerId(answer_id: String?) = apply { this.answer_id = answer_id }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(update_immediately: Boolean) =
            apply { this.update_immediately = update_immediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw and exception if the questionId provided is null or no value is
                 * provided at all.
                 */
                question_id.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        ELearnEventType.question.name,
                        "question_id"
                    )
                }

                /**
                 * Will throw and exception if the action provided is null or no value is
                 * provided at all.
                 */
                action.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        ELearnEventType.question.name,
                        QuestionAction::class.java.simpleName
                    )
                }

                /**
                 * Will throw and exception if the examId provided is null or no value is
                 * provided at all.
                 */
                exam_id.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        ELearnEventType.question.name,
                        "exam_id"
                    )
                }

                else -> {
                    /**
                     * Will throw and exception if the answerId provided is null or no value is
                     * provided at all.
                     */
                    if (action == QuestionAction.answer.name && answer_id.isNullOrEmpty()) {
                        ExceptionManager.throwIsRequiredException(
                            ELearnEventType.question.name,
                            "answer_id"
                        )
                    } else {
                        /**
                         * Parsing the values into an object and passing to the setup block to queue
                         * the event based on its priority.
                         */
                        val questionObject = QuestionObject(
                            question_id!!, exam_id!!, action!!,
                            answer_id, meta
                        )
                        CFSetup().track(
                            ELearningConstants.contentBlockName, ELearnEventType.question.name,
                            questionObject, update_immediately
                        )
                    }
                }
            }
        }
    }
}
