package ai.causalfoundry.android.sdk.e_learning.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_learning.event_models.event_objects.QuestionObject
import ai.causalfoundry.android.sdk.e_learning.event_types.ELearnEventType
import ai.causalfoundry.android.sdk.e_learning.event_types.QuestionAction
import ai.causalfoundry.android.sdk.e_learning.utils.ELearningConstants

/**
 * Created by Moiz Hassan on 23 March, 2023
 */

class CfLogQuestionEvent {

    /**
     * CfLogQuestionEvent is required to log user answers to the questions. To log this event you
     * need to provide the question Id that User has attempted and also the id for the answer the
     * user has selected.
     */

    data class Builder(
        var questionIdValue: String? = null,
        var examIdValue: String? = null,
        var actionValue: String? = null,
        var answerIdValue: String? = null,
        private var meta: Any? = null,
        private var isUpdateImmediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * setQuestionId is required to log questionId for the Question on which user has attempted.
         * Question Id should be in a string format and must be in accordance to the catalog
         * provided.
         */
        fun setQuestionId(questionId: String) = apply { this.questionIdValue = questionId }

        /**
         * setExamId is required to log exam_id for the exam the Question belongs to.
         * exam_id should be in a string format and must be in accordance to the catalog
         * provided.
         */
        fun setExamId(examId: String) = apply { this.examIdValue = examId }

        /**
         * setQuestionAction is required to set the Action type for the Question event. SDK provides
         * enum classes to support available log types. 2 main are answer and skip.
         * SDK provides 2 approaches to log this event, one being enum type and the other is
         * string type. Below is the function to log type using enum.
         */
        fun setQuestionAction(action: QuestionAction) = apply { this.actionValue = action.name }

        /**
         * setQuestionAction is required to set the Action type for the Question event. SDK provides
         * enum classes to support available log types. 2 main are answer and skip.
         * SDK provides 2 approaches to log this event, one being enum type and the other is
         * string type. Below is the function to log type using string. Remember to note that
         * values provided using string should be the same as provided in enum or else the
         * events will be discarded.
         */
        fun setQuestionAction(action: String?) = apply {
            if (action != null) {
                if (CoreConstants.enumContains<QuestionAction>(action)) {
                    this.actionValue = action
                } else {
                    ExceptionManager.throwEnumException(
                        ELearnEventType.question.name,
                        QuestionAction::class.java.simpleName
                    )
                }
            } else {
                this.actionValue = action
            }
        }

        /**
         * setAnswerId is required to log answerId for the answer provided by the user for
         * the Question on which user has attempted. Answer Id should be in a string format and
         * must be in accordance to the catalog provided.
         */
        fun setAnswerId(answerId: String?) = apply { this.answerIdValue = answerId }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(updateImmediately: Boolean) =
            apply { this.isUpdateImmediately = updateImmediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw and exception if the questionId provided is null or no value is
                 * provided at all.
                 */
                questionIdValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        ELearnEventType.question.name,
                        "question_id"
                    )
                }

                /**
                 * Will throw and exception if the action provided is null or no value is
                 * provided at all.
                 */
                actionValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        ELearnEventType.question.name,
                        QuestionAction::class.java.simpleName
                    )
                }

                /**
                 * Will throw and exception if the examId provided is null or no value is
                 * provided at all.
                 */
                examIdValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        ELearnEventType.question.name,
                        "exam_id"
                    )
                }

                else -> {
                    /**
                     * Will throw and exception if the answerId provided is null or no value is
                     * provided at all.
                     */
                    if (actionValue == QuestionAction.answer.name && answerIdValue.isNullOrEmpty()) {
                        ExceptionManager.throwIsRequiredException(
                            ELearnEventType.question.name,
                            "answer_id"
                        )
                    } else {
                        /**
                         * Parsing the values into an object and passing to the setup block to queue
                         * the event based on its priority.
                         */
                        val questionObject = QuestionObject(
                            questionIdValue!!, examIdValue!!, actionValue!!,
                            answerIdValue, meta
                        )
                        CFSetup().track(
                            ELearningConstants.contentBlockName, ELearnEventType.question.name,
                            questionObject, isUpdateImmediately
                        )
                    }
                }
            }
        }
    }
}
