package ai.causalfoundry.android.sdk.e_learning.builders

import ai.causalfoundry.android.sdk.core.CFSetup
import ai.causalfoundry.android.sdk.core.utils.CoreConstants
import ai.causalfoundry.android.sdk.core.utils.ExceptionManager
import ai.causalfoundry.android.sdk.e_learning.event_models.event_objects.ExamObject
import ai.causalfoundry.android.sdk.e_learning.event_types.ELearnEventType
import ai.causalfoundry.android.sdk.e_learning.event_types.ExamAction
import ai.causalfoundry.android.sdk.e_learning.utils.ELearningConstants


/**
 * Created by Moiz Hassan on 23 March, 2023
 */

class CfLogExamEvent {

    /**
     * CfLogExamEvent is required to log actions related to e-learning module exams. which
     * includes the related to starting, retaking, reviewing and submit the exam. BsLogExamEvent
     * also updates the user level if they achieved a milestone.
     */
    data class Builder(
        var examIdValue: String? = null,
        var actionValue: String? = null,
        var durationValue: Int? = null,
        var scoreValue: Float? = null,
        var isPassedValue: Boolean? = null,
        private var meta: Any? = null,
        private var isUpdateImmediately: Boolean = CoreConstants.updateImmediately
    ) {

        /**
         * setExamId is required to log examId for the Exam on which user is performing actions.
         * Exam Id should be in a string format and must be in accordance to the catalog
         * provided.
         */
        fun setExamId(examId: String) = apply { this.examIdValue = examId }

        /**
         * setExamAction is required to set the Action type for the Exam event. SDK provides
         * enum classes to support available log types. 3 main are start, submit and result.
         * SDK provides 2 approaches to log this event, one being enum type and the other is
         * string type. Below is the function to log type using enum.
         */
        fun setExamAction(action: ExamAction) = apply { this.actionValue = action.toString() }

        /**
         * setExamAction is required to set the Action type for the Exam event. SDK provides
         * enum classes to support available log types. 3 main are start, submit and result.
         * SDK provides 2 approaches to log this event, one being enum type and the other is
         * string type. Below is the function to log type using string. Remember to note that
         * values provided using string should be the same as provided in enum or else the
         * events will be discarded.
         */
        fun setExamAction(action: String) = apply {
            this.actionValue = action
        }

        /**
         * setDuration is required to log the duration (time elapsed) by the user to complete the
         * exam. Duration should be in Seconds. This is required in case of examAction been submit.
         */
        fun setDuration(duration: Int?) = apply { this.durationValue = duration }

        /**
         * setScore is required if there is some score provided ot the user in result of the
         * exam submitted. This is required in case of examAction been result.
         */
        fun setScore(score: Float?) = apply { this.scoreValue = score }

        /**
         * isPassed is required if the user passed or failed the exam. This log is required only
         * in case when examAction is result
         */
        fun isPassed(isPassed: Boolean?) = apply { this.isPassedValue = isPassed }

        /**
         * You can pass any type of value in setMeta. It is for developer and partners to log
         * additional information with the log that they find would be helpful for logging and
         * providing more context to the log. Default value for the meta is null.
         */
        fun setMeta(meta: Any?) = apply { this.meta = meta }

        /**
         * updateImmediately is responsible for updating the values ot the backend immediately.
         * By default this is set to false or whatever the developer has set in the SDK
         * initialisation block. This differs the time for which the logs will be logged, if true,
         * the SDK will log the content instantly and if false it will wait till the end of user
         * session which is whenever the app goes into background.
         */
        fun updateImmediately(updateImmediately: Boolean) =
            apply { this.isUpdateImmediately = updateImmediately }

        /**
         * build will validate all of the values provided and if passes will call the track
         * function and queue the events based on it's updateImmediately value and also on the
         * user's network resources.
         */
        fun build() = apply {
            when {

                /**
                 * Will throw and exception if the examId provided is null or no value is
                 * provided at all.
                 */
                examIdValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(ELearnEventType.Exam.toString(), "exam_id")
                }

                /**
                 * Will throw and exception if the action provided is null or no value is
                 * provided at all.
                 */
                actionValue.isNullOrEmpty() -> {
                    ExceptionManager.throwIsRequiredException(
                        ELearnEventType.Exam.toString(),
                        ExamAction::class.java.simpleName
                    )
                }

                !CoreConstants.enumContainsString<ExamAction>(actionValue!!) -> {
                    ExceptionManager.throwEnumException(
                        ELearnEventType.Exam.toString(),
                        ExamAction::class.java.simpleName
                    )
                }

                else -> {
                    if (actionValue == ExamAction.Submit.toString() && durationValue == null) {
                        ExceptionManager.throwIsRequiredException(
                            ELearnEventType.Exam.toString(),
                            "duration_value"
                        )
                    } else if (actionValue == ExamAction.Result.toString() && scoreValue == null) {
                        ExceptionManager.throwIsRequiredException(
                            ELearnEventType.Exam.toString(),
                            "score"
                        )
                    } else if (actionValue == ExamAction.Result.toString() && isPassedValue == null) {
                        ExceptionManager.throwIsRequiredException(
                            ELearnEventType.Exam.toString(),
                            "is_passed"
                        )
                    } else {
                        /**
                         * Parsing the values into an object and passing to the setup block to queue
                         * the event based on its priority.
                         */
                        val examObject = ExamObject(
                            examIdValue!!,
                            actionValue!!,
                            durationValue,
                            scoreValue,
                            isPassedValue,
                            meta
                        )
                        CFSetup().track(
                            ELearningConstants.contentBlockName, ELearnEventType.Exam.toString(),
                            examObject, isUpdateImmediately
                        )
                    }
                }
            }
        }
    }
}
